

;***************************************************************************
; 80W Solar Charger (based on core idea from SilChip MPPT Charger 2016)
; recoded for 16f1847 (uses same pinout as 16f88)
;***************************************************************************

; $e32a 0v2e 28mar2022



;***************************************************************************
; OPERATION
;***************************************************************************
;SCAN_SWITCH
;	short press = scroll thru menu
;	medium press toggle LOAD (LOAD defaults to ON if >12.5v, but can be manually turned off)
;	long press = REBOOT
;	press during BOOT = Enable EQ charging (displays on bottom LHS of Menu1 screen when enabled)
;
;CHARGE MODES
; ABS (& optionally EQ) enabled once perday, but currently limited to 30 minutes runtime
; BULK goes direct to FLOAT if reaches end in <60secs
; DUSK State entered immediately on NOSUN, or after 30Mins LOWSUN. Thereafter it enters SLEEP state.
; SLEEP State (display off) exited on return of FULL SUN or a loooong switch press (full REBOOT)
; DUSK mode display is simple, just a timer, temperature and BattV display
; LOAD OFF until dusk, when it auto turns ON for night lights (until sleep)

; Solar Charger (based on core idea from SilChip MPPT Charger 2016)
;****************************************************************************
;* 16f1847                                                                   *
;*                             --------uu--------                           *
;*                solar_V --->| ra2 an2  an1 ra1 |<--- load_I               *
;*                solar_I --->| ra3 an3  an0 ra0 |<--- battery_V            *
;*          temperature_V --->| ra4 an4      ra7 |---> OPTO_ON (active HI)  *
;*                  /MCLR --->| ra5          ra6 |---> load_ON (active HI)  *
;*                     0v ----| vss          vdd |<--- +5V                  *
;*                LCD_/EN <---| rb0      pgC rb7 |---> LCD_D7               *
;*                LCD_/RS <---| rb1      pgD rb6 |---> LCD_D6               *
;*     switch (active HI) --->| rb2          rb5 |---> LCD_D5               *
;*                pwm out <---| rb3 pwm      rb4 |---> LCD_D4               *
;*                             ------------------                           *
;*                                                                          *
;***************************************************************************

	list P=16f1847
	#include p16f1847.inc

	errorlevel -302		; Suppress "Not in Bank 0" warning messages
	errorlevel -305		; Suppress "Using default destination of 1 (file)" warning messages
	errorlevel -306		; Suppress "Crossing page boundary=ensure page select set correctly" warning messages
	errorlevel -205		; Suppress "Found Directive in Column 1" warning messages

;NOTE WDT must be off here, as it is under WDTCON software control.
    __CONFIG _CONFIG1, _FOSC_INTOSC & _WDTE_SWDTEN & _PWRTE_ON & _MCLRE_ON & _CP_OFF & _CPD_OFF & _BOREN_OFF & _CLKOUTEN_OFF & _IESO_OFF & _FCMEN_OFF
    __CONFIG _CONFIG2, _WRT_OFF & _PLLEN_OFF & _STVREN_OFF & _BORV_HI & _LVP_OFF


; ********* define terms ********* 
#DEFINE ZEROBIT STATUS,2	; = shorthand for STATUS zero flag or "STATUS,Z"
#DEFINE CARRYBIT STATUS,0	; = shorthand for STATUS CARRY flag or "STATUS,C"
#DEFINE CLRCARRY bcf STATUS,0	; = shorthand for clear CARRY flag

#DEFINE EN_PIN PORTB,0
#DEFINE RS_PIN PORTB,1
#DEFINE SW_PIN PORTB,2
#DEFINE PWM_PIN PORTB,3
#DEFINE D4_PIN PORTB,4
#DEFINE D5_PIN PORTB,5
#DEFINE D6_PIN PORTB,6
#DEFINE D7_PIN PORTB,7
; #DEFINE AD0_IN PORTA,0 ; (battery volts)
; #DEFINE AD1_IN PORTA,1 ; (load current)
; #DEFINE AD2_IN PORTA,2 ; (solar volts)
; #DEFINE AD3_IN PORTA,3 ; (solar current)
; #DEFINE AD4_IN PORTA,4 ; (temperature sensor volts)
; #DEFINE MCLR	PORTA,5
#DEFINE LOAD_ON	 bsf PORTA,6	; or use macro instead
#DEFINE LOAD_OFF bcf PORTA,6	; or use macro instead
#DEFINE OPTO_ON  bsf PORTA,7
#DEFINE OPTO_OFF bcf PORTA,7

; error & trip thresholds. these are LO byte or 8-bit values
#DEFINE CHG_I_LIMIT d'38'	; Battery moderate C/8 charge limit (assuming 12AH battery)
#DEFINE DUSK_MAX  d'180'	; Max #minutes allowed in DUSK mode before entering SLEEP mode

#DEFINE TV_MIN	h'2f'	; Temperature Voltage - LO byte of $022f = 0Celcius
#DEFINE TV_MAX	h'81'	; Temperature Voltage - LO byte of $0281 = 40Celcius	
#DEFINE KV20_LO	h'58'	; TempVolts in Kelvin - LO byte of 293K = 20C(Vcomp. cut-in point)
#DEFINE SV_FAIL	h'09'	; Solar Volts - <1V  8-bit value (when fail message shown)
#DEFINE SV_LOW	h'70'	; Solar Volts - <12.5V  8-bit value (when charging stops)	
#DEFINE BV_MIN	h'86'	; Battery Volts - <9.5v  8-bit value
#DEFINE BV_MAX	h'D8'	; Battery Volts >15v 8-bit value

;Battery voltage charge-point thresholds (assumes Vcc = 5.00V)
#DEFINE BST_ENTRY	h'88'	; 9.5v
#DEFINE BST_EXIT	h'97'	; 10.5v

#DEFINE TKL_ENTRY	h'97'	; 10.5v
#DEFINE TKL_EXIT	h'b4'	; 12.55v
#DEFINE TKL_FAIL	h'90'	; 10v

#DEFINE BLK_ENTRY	h'B4'	; 12.5v
#DEFINE BLK_EXIT_H	h'03'	; 13.7v (2 byte 10 bit value)
#DEFINE BLK_EXIT_L	h'15'
#DEFINE BLK_FAIL	h'AC'	; 12v

#DEFINE ABS_HOLD_H	h'03'	; 13.7v (2 byte 10 bit value)
#DEFINE ABS_HOLD_L	h'15'	
;#DEFINE ABS_EXIT		; not needed, ABS uses 1 Hour timer to exit 
#DEFINE ABS_FAIL	h'BB'	; 12.5v (currently same value as used by EQUALISE & FLOAT)

#DEFINE EQL_HOLD_H	h'03'	; (should be 14.49v, but left 13v7 for safety while testing- same as ABS)
#DEFINE EQL_HOLD_L	h'15'	; 

#DEFINE FLT_ENTRY	h'Be'	; 13.2v
#DEFINE FLT_HOLD_H	h'03'	; 13.7v (2 byte 10 bit value)
#DEFINE FLT_HOLD_L	h'15'
#DEFINE FLT_FAIL	h'bb'	; 13v 

; A/D reading conversion factors 
#DEFINE BV_LO	h'de'	; battery volts 
#DEFINE BI_LO	h'3e'	; battery current = 0.1R = 5A max (or $7d for 0.05R = 10A max);

#DEFINE SV_HI	h'01'	; solar volts $0163
#DEFINE SV_LO	h'63'	;
#DEFINE SI_LO	h'3e'	; solar current = 0.1R = 5A max, (or $7d for 0.05R = 10A max)

#DEFINE SP_HI	h'03'	; solar power $0378 = 5A x 28.4V = 142W max, (or $06ef for 10A x 28.4V = 284W max)
#DEFINE SP_LO	h'78'	;
 
#DEFINE KV_HI	h'02'	; $0271 - (HI BYTE NOT NEEDED)
#DEFINE KV_LO	h'71'	; temperature in KELVIN

#DEFINE TCA_HI	h'af'	; temperature comp. adjustment in KELVIN (for battery)
#DEFINE TCA_LO	h'e5'	;
#DEFINE TC_HI	h'11'	; temperature comp. display in mV/degree CELCIUS
#DEFINE TC_LO	h'59'	;

; ********* define constants *********


; ********* define variables ********* 
	cblock	0x20 ; (Bank0) cblock & endc autoallocate RAM space, no need for EQUate statement
CNTR_HBEAT	; increments each INT = 0.262Sec 
CNTR_MINS	; General purpose 60.3Sec timer run by INT
CNTR_HRS	; 4Hr+ counter, 1 minute per bit
CNTR_DUSK	; sleep-mode counter

CCPR1L_STORE	; PWM setting, CCPR1L temporary storage
PWR_LO		; power ls byte
PWR_HI		; power ms byte

; ADC variables
ADC_WAIT	; A/D sample time count
BATT_V_LO	; battery volts low byte
BATT_V_HI	; battery volts high byte
BATT_V_8BIT	; battery volts volts 8-bit
BATT_I_LO	; load current low byte
BATT_I_HI	; load current high byte
BATT_I_8BIT	; battery load current 8-bit
SOLAR_V_LO	; solar cell voltage ls byte
SOLAR_V_HI	; solar cell voltage ms byte
SOLAR_V_8BIT	; solar cell volts 8-bit
SOLAR_I_LO	; solar cell current ls byte
SOLAR_I_HI	; solar cell current ms byte
SOLAR_I_8BIT	; solar cell current 8-bit
TEMP_V_LO	; temperature reading - 10mV/degC kelvin
TEMP_V_HI	; temperature reading - 10mV/degC kelvin
TEMP_V_8BIT	; temperature (kelvin) volts 8-bit

; misc variables
TCOMP_VAL	; temp compensated V adjustment (10bit res = 17.61mV per unit)
CELCIUS		; temperature in celcius 

THRESH_LO	; V_threshold for current charge mode, LO byte
THRESH_HI	; V_threshold for current charge mode, HI byte
THRESH_LO_ADJ	; temperature adjusted V_threshold, LO byte
THRESH_HI_ADJ	; temperature adjusted V_threshold, HI byte

TIME_INIT	; delay timer - initial multiplier value
TIME1		; delay timer delay store 1 
TIME2		; delay timer delay store 2 
BIN_INPUT	; used in debug routines
HEX1_OUT	; used in debug routines
HEX2_OUT	; used in debug routines
;TEMPERATURE	; reseerved for NTC thermister option

; math routine variables
TEMP
TEMP1	
TEMPB0	
TEMPB1	
TEMPB2	
REMB3	
REMB2	
REMB1    
REMB0	
AARGB5	
AARGB4    
AARGB3	
AARGB2    
AARGB1     
AARGB0		; msb of argument A
BARGB3      
BARGB2      
BARGB1      
BARGB0		; msb of argument B
LOOPCOUNT	; division counter

;used by BIN_BCD binary to ASCII conversion
BCD1		; overflow 
BCD2		; msb
BCD3
BCD4
BCD5		; lsb
COUNT
EEIGHT		; hex ascii of BCD (overflow)
ESEVN		; msb hexascii digit pair
ESIX
EFIVE		; hexascii digit pair
EFOUR
ETHREE		; hexascii digit pair
ETWO
EONE		; lsb hexascii digit pair
EZERO
	endc
	
	cblock	0x70	; $70-$7F are visible to all Banks of RAM
TMPVAL_LO	; temporary value ls byte, used in ADC & PWR calcs
TMPVAL_HI	; temporary value ms byte, used in ADC & PWR calcs
TMPVAL_8BIT	; temporary 8bit ADC result

TEMPSTORE	; temporary storage, mainly debug routines

STRING_PTR	; Relative Pointer for current Menu Text
TABLE_OFFSET	; offset above "Table" start address for STROUT txt string

LCD_BSY		; busy time delay for display
LCD_DATA	; TEMP LCD data storage

MMI_FLAG	; new_keypress<7>, refreshMMI<6>, unused<5>, chr_toggle<4>, menu3<3>, menu2<2> menu1<1> menu0<0>
CHG_FLAG	; EQ_Yes<7>,ABS_Yes<6>,[Float]<5>,[Eq]<4>,[Absrb]<3>,[Bulk]<2>,[Trkl]<1>,Sleep<0>
ERR_FLAG	; unused<7>, noTemp<6>, >40C<5>, <0C<4>, >15V<3>, <9V<2>, NoSUN<1>, LowSUN<0>
MPPT_FLAG	; scan running <7>, scan due<6>,tracking<5>,<4>,<3>,<2>,PWR increased<1>,inc=1 dec=0<0>
	endc
	
; ********* define macros ********* 
multiply_2424	macro	; Call subroutine in page 1 (800h-FFFh)
	bcf	INTCON,GIE	; disable global interrupts
	pagesel	FXM2424U	; or 0x0800 or PAGE1CALL
	call	FXM2424U	; multiply!
	pagesel	BEGIN		; or 0x0000 or PAGE1RET
	bsf	INTCON,GIE	; enable global interrupts
	endm		

multiply_8bit	macro	; Call subroutine in page 1 (800h-FFFh)
	bcf	INTCON,GIE	; disable global interrupts
	pagesel	EIGHTEIGHT	; PAGE1CALL
	call	EIGHTEIGHT	; multiply!
	pagesel	BEGIN		; or 0x0000 or PAGE1RET
	bsf	INTCON,GIE	; enable global interrupts
	endm

divide_3232	macro	; Call subroutine in page 1 (800h-FFFh)
	bcf	INTCON,GIE	; disable global interrupts
	pagesel	FXD3232U	; PAGE1CALL
	call	FXD3232U	; divide!
	pagesel	BEGIN		; or 0x0000 or PAGE1RET
	bsf	INTCON,GIE	; enable global interrupts
	endm

binary_2_bcd	macro	; Call subroutine in page 1 (800h-FFFh)
	bcf	INTCON,GIE	; disable global interrupts
	pagesel	BIN_BCD		; PAGE1CALL
	call	BIN_BCD		; convert binary to BCD
	pagesel	BEGIN		; or 0x0000 or PAGE1RET
	bsf	INTCON,GIE	; enable global interrupts
	endm

; compare 16bit X with Y threshold, 
compare_16	macro	x_hi, x_lo, y_hi, y_lo
local	HI_NEQ			; local macro label definition
	movf	x_hi,w		; X_Hi
	subwf	y_hi,w		; w = Y_Hi(file/threshold) - X_Hi(w) 
	btfss	ZEROBIT		; equal?
	goto	HI_NEQ		; no, goto HIbyte_NOT_EQ
	movf	x_lo,w		; yes equal. so get new X (LO byte)
	subwf	y_lo,w		; w = Y_Lo(file) - X_Lo(w)
HI_NEQ	btfss	CARRYBIT	; HIbytes NOT equal, so is borrow needed? ie Y=>X?
	goto	$+2		; goto X_GREATER (as no borrow was required)
; ...in Main code, add a goto HERE if Y => X
; or in Main code, add a goto HERE if X > Y
	endm

subtract_16	macro   x_hi, x_lo, y_hi, y_lo
; 16-bit unsigned subtraction. X-Y=X (X becomes result, Y is preserved, w = LO byte of difference)
	movf    y_lo,w		; get lo-byte of subtrahend
	subwf   x_lo,f		; subtract x(low) - y(low)
	movf    y_hi,w		; get high byte of subtrahend
	btfss   CARRYBIT	; if there was a borrow, rather than
	incf    y_hi,w		; decrement hi-byte of X we inc Y
	subwf   x_hi,f		; then subtract x(hi) - y(hi)
	endm   

battlow_chk	macro	voltage_thrshld	; All Charge Modes come here to check if Batt-V suffient for current Mode
	movfw	BATT_V_8BIT	; is battery < current voltage_thrshld?
	sublw	voltage_thrshld	; Literal(threshold) - w(Battery) = w
	btfsc	CARRYBIT	; CARRY clear if w(Battery) > threshold
	goto	CHG_MODE_SELECT	; CARRY SET: Battery too low, reselect mode
;else in Maincode,continue HERE	; CARRY CLR
	endm

errorflag_chk	macro		; ALL Charge Modes visit here, check FLAGS and change Mode if required!
	movf	ERR_FLAG,w	; unused<7>, noTemp<6>, >40C<5>, <0C<4>, >15V<3>, <9V<2>, NoSUN<1>, LowSUN<0>	
	andlw	b'01111100'	; check all Error FLAGS, except NoSUN<2>, SunLO<0>
	btfss	ZEROBIT
	goto	FAULT_STATE	; CARRY SET: error detected; analyse!
;else in Maincode,continue HERE	; CARRY CLR
	endm

duskflag_chk	macro		; ALL charging states periodically check here to see if DUSK_STATE warranted
	btfsc	ERR_FLAG,1	; is NoSUN set?
	goto	$+5		; yes, so check DUSK timer
	btfss	ERR_FLAG,0	; is LowSUN set?
 	goto	$+5		; no, just exit
	btfss	CNTR_DUSK,5	; has 32 minutes passed?
	goto	$+3		; not yet time, so just exit
; time to enter DUSK mode
	bsf	CHG_FLAG,0	; set SLEEP flag ...
	goto	DUSK_STATE	; ...and enter DUSK state
;else in Maincode,continue HERE	; flag no set	
	endm			

show_load_state	macro		; used by Menu display #4 and Error display.
	movlw	'L'
	call	LCD_CHR
	movlw	'o'
	call	LCD_CHR
	movlw	'a'
	call	LCD_CHR
	movlw	'd'
	call	LCD_CHR
; show l ON or OFF
	btfss	PORTA,6
	goto	$+3
	call	SHOW_ON
	goto	$+2
	call	SHOW_OFF
	movlw	' '
	call	LCD_CHR
	endm

; Most PWM/CCPR1L adjustments put into macros as EMR device Bank swaps makes code look messy!
clear_PWM	macro		; PWM adjustments look cleaner without messy Bank swaps!
	banksel	CCPR1L		; Bank5
	clrf	CCPR1L		; stop PWM output
	banksel	d'0'		; Bank0
	endm

read_PWM	macro		; PWM adjustments look cleaner without messy Bank swaps!
	banksel	CCPR1L		; Bank5
	movf	CCPR1L,w	; load W from PWM
	banksel	d'0'		; Bank0
	endm

write_PWM	macro		; PWM adjustments look cleaner without messy Bank swaps!
	banksel	CCPR1L		; Bank5
	movwf	CCPR1L		; save W to PWM
	banksel	d'0'		; Bank0
	endm

inc_PWM		macro		; PWM macros save a lot of Bank swapping in code!
	banksel	CCPR1L		; Bank5
	incf	CCPR1L,f	; increase PWM
	banksel	d'0'		; Bank0
	endm

dec_PWM		macro		; PWM adjustments look cleaner without messy Bank swaps!
	banksel	CCPR1L		; Bank5
	decf	CCPR1L,w	; decrease PWM
	banksel	d'0'		; Bank0
	endm

; The 16F1847 has 256 bytes of non-volatile EEPROM, starting at address 0xF000
;------------------------------------------------------------------------------
	ORG 0xF000	; preprogram EEPROM DATA 
	DE	"MCHP"	; Place 'M' 'C' 'H' 'P' at address 0,1,2,3
	ORG 0xF010
	DE	'c', 'P', 'L', 'N','2', '0', '2', '1'


; ********************************************************************************
; start at memory 0
; ********************************************************************************
BEGIN:	org	0		; processor reset vector, start at memory 0
	goto	SETUP
	org	4		; interrupt vector 0004h
	goto	INTERRUPT


; ********************************************************************************
INTERRUPT:	 ; cycles every 0.262Seconds
; INT source = TMR1 (16 bit counter) 
; T1CON period = [8mhz] Fosc/4 x 8 prescaler x 2^16 = [125uS x 4] x 8 x 65536 = 262.144mS 
; ********************************************************************************	
;Note: GIE bit is auto-cleared on Int Entry and auto-reset on Int Exit
;Note: EMR devices auto-saves/restores status registers, so no pop/pull code needed! 
	banksel	PIR1		; Bank 0	
	bcf	PIR1,TMR1IF	; clear (only) INTerrupt source
	clrwdt			; clear Watchdog

; ---------------------------------------
; HEARTBEAT:	; Interrupt cycle is 0.262Secs, provides basis for CLOCK, SLEEP and  LCD refresh.
; ---------------------------------------
	incf	CNTR_HBEAT	; increment INT heartbeat...
; ...and periodically force a LCD update
	movf	CNTR_HBEAT,w	; AND with last 3 bits only (= 8 x 0.262 = 2.097Sec)
;	andlw	B'00000111'	; and when count = xxxxx000, another 2 seconds has elapsed
	andlw	B'00000011'	; and when count = xxxxx000, another 1 second has elapsed
 	btfsc	ZEROBIT		; Zerobit set? (ie OVER time threshold mask?)
	bsf	MMI_FLAG,6	; Zero set. So set MMI_FLAG,6 to tell state machines to update LCD

; ---------------------------------------
;CHK_MINUTES: ; increments every 229 x 0.262S = 60Sec
; ---------------------------------------
	movf	CNTR_HBEAT,w
	sublw	d'230'		; rollover = (0.262Sec x 230)= 60.293 seconds
	btfsc	CARRYBIT	; CARRY set?
	goto	CLOCK_DONE	; No, exit
; Yes, a minute has elapsed...
	clrf	CNTR_HBEAT	; Yes, restart HEARTBEAT counter
	incf	CNTR_MINS	; Inc minutes...
	incf	CNTR_DUSK	; ...inc Low/NoSUN timer, counts to 256Mins (reset by Solar_ADC LOW_SUN detection)
	bsf	MPPT_FLAG,6	; & set FLAG for a new MAX POWER run (only cleared after a MPPT PWM scan)
; ---------------------------------------
;CHK_HOURS:
; ---------------------------------------
	movf	CNTR_MINS,w	; get Minutes count 
	sublw	d'60'
	btfsc	CARRYBIT	; 60Mins just rolled over?
	goto	CLOCK_DONE	; No, exit
; Yes, an hour has elapsed...
	incf	CNTR_HRS	; Yes, so inc Hour count...
	clrf	CNTR_MINS	; ...and restart minutes count
CLOCK_DONE:

; ---------------------------------------
; CHK_DUSK:	SLEEP counter runs when solarV too low to charge (otherwise always cleared)
; ---------------------------------------
	movf	ERR_FLAG,w	; get ERROR flags
	andlw	b'00000011'	; mask off all but ERR_FLAG  NoSUN<1>, LowSUN<0>
	btfsc	ZEROBIT	
	clrf	CNTR_DUSK	; disable counting while SolarV is capable of charging
DUSK_DONE:
INT_EXIT
	retfie			; (RETFIE auto re-enables GIE bit)


; ********************************************************************************
SETUP:
; *********************************************************************************
; set WDT only after initialising LCD (as LCD takes a LOOONG time)
	banksel	WDTCON
;	movlw	b'00010110'	; S/W WDT (31.5khz INTRC clk/65556) = 2.09S timeout
;	movwf	WDTCON		; WDT prescaler (WDT enabled in config word)
	bcf	WDTCON,0	; SW DISable WDT	

; set inputs/outputs
	banksel	OSCCON		; Bank1
	movlw	b'01110010'	; 8MHz, running from INTRC
	movwf	OSCCON		; set clock
	movlw	b'00000000'
	movwf	OSCTUNE		; osc offset

	banksel	CM1CON0		; Bank2
	movlw	b'00000000'	; comparator1 off
	movwf	CM1CON0		; comparator1
	movlw	b'00000000'	; comparator2 off
	movwf	CM2CON0		; comparator2

; set portA & B data direction, pullups, + analogue inputs.
	banksel	LATA		;
	clrf	LATA		;Data Latch PORTA
	clrf	LATB		;Data Latch PORTB

	banksel	WPUA
	clrf	WPUA		; disable all WPU (not good for A/D channels)
	clrf	WPUB		; disable all WPU (not needed on Act-Hi SW input RB2)

; set port data direction, where (0)=outputs and (1)=inputs
	banksel	TRISA		; Bank1 - DataDir register
	movlw	b'00111111'	; OPTO_ON<7>, LOAD_ON<6>, all else inputs
	movwf	TRISA
	movlw	b'00000100'	; SW_IN<2>, all else outputs
	movwf	TRISB

	banksel	OPTION_REG	; Bank1
	movlw	b'10000111'	; WPU pullups DISabled, TMR0/256 prescaler
	movwf	OPTION_REG

; setup & enable A/D
; ...................................................
; optionally, use FVR as Vdd +ve reference for A/D?
	banksel	FVRCON
	movlw	b'10000011'	; set FVR ENabled & ADC buffer gain x4
	movwf	FVRCON		
; ...................................................
	banksel	ANSELA		; Bank3
	movlw	b'00011111'	; setup PortA <4:0> as analog inputs
	movwf	ANSELA
	clrf	ANSELB		; (portB currently not used for any analogue)

	banksel	ADCON0		; Bank1
	movlw	b'00000000'	; default is chan select 000 (AN0) & set ADC to off
	movwf	ADCON0
	movlw	b'11010000'	; right justified result,(Fosc/16) = 2uS AD clk source),  A/D REF = Vss-Vdd
;	movlw	b'11010011'	; right justified result,(Fosc/16) = 2uS AD clk source),  A/D REF = Vss-Vfvr (4.096v)
	movwf	ADCON1
	bsf	ADCON0,ADON	; A/D converter enabled!

; timer 1
	banksel	T1CON		; Bank0
	movlw	b'00110001'	; [if 8MHz] Fosc/4 SysClk, prescaler/8, no ext sync, TMR1 ON	
	movwf	T1CON
	bsf	T1CON,TMR1ON	; timer 1 enabled

; set pwm
; ***************************************************
; PWM period = (PR2+1) x 4 x Tosc x T2prescale = (63+1) x0.125uS x4 x 1 = 64 x 0.5uS = 32uS
; frequency  = 1/period =31.25kHz
; dutycycle  = CCPRL1:CC1CON<5,4> x Tosc x T2prescale = (63 x 4) x 0.125uS x 1 = 0(disabled) = 31.5uS(98.5%)
	banksel	PR2		; Bank0
	movlw	H'3F'		; = 31.25kHz pwm rate (8MHz clock)
	movwf	PR2		; set PWM period register

	banksel	CCPR1L		; Bank5
	clrf	CCPR1L		; set PWM DutyCycle to 100% OFF
	movlw	b'00001100'	; enable PWM, 8-bit resolution (bits 4&5 unused = '00')
; Must be in 8bit mode (not10bit) else the PWM_OFF command will NOT totally kill PWM output!
	movwf	CCP1CON		; enable PWM mode
	clrf	CCPTMRS		; set TMR2 as source timebase (for all 4 PWMS's)

	banksel	T2CON		; Bank0
	clrf	T2CON
	bsf	T2CON,TMR2ON	; enable Timer2
; initialise values
; ***************************************************
	clrf	PORTB
	clrf	PORTA

	clrf	CNTR_HBEAT	; general purpose 60.3Sec counter
	clrf	CNTR_MINS	; general purpose Minutes counter
	clrf	CNTR_HRS	; general purpose Hours counter
	clrf	CNTR_DUSK	; SLEEP minutes counter (256 x 1 minute = 4.26Hrs max possible)

	clrf	CCPR1L_STORE	; CCPR1L storage (MSB of duty cycle)
	clrf	MPPT_FLAG	; first run
	clrf	ERR_FLAG	; no errors as yet

	clrf	PWR_HI		; power ms byte
	clrf	PWR_LO		; power ls byte
;------------------------------------------------
; LCD & switch additions
	OPTO_ON			; enable LCD and OPTO
	call	DELAY1mS	; wait for power to stabilise...

; enable LOAD_ON unless manually switched off OR if battery <12v5
	LOAD_ON			; backlite on for splash screen - manual selection thereafter

	clrf	MMI_FLAG	; no switch presses yet.
	bsf	MMI_FLAG,0	; enable default display

	clrf	CHG_FLAG	; EQ_Yes<7>,ABS_Yes<6>,unusedt<5><4><3><2>,Trkl<1>,Sleep<0>
	bsf	CHG_FLAG,6	; ABSmode is allowed once per day (but EQ is not)

	call	INIT_DISPLAY	; init LCD in 4bit mode
	call	SPLASH		; splash screen
	movlw	d'100'		; show for ~50x10mS = 0.5S so user is aware...
	call	DELAYx10

; ***************************************************
; enable EQUALISATION if switch is pressed & held during boot
; ***************************************************
	btfss	SW_PIN		; switch pressed?
	goto	EQ_DONE_DELAY	; no, but exit with a delay so as SPLASH is visible
; a switch was pressed, but check its not a glitch
	movlw	d'10'		; ~ 100mS
	call	DELAYx10	; Debounce line
	btfss	SW_PIN		; still pressed?
	goto	EQ_DONE_DELAY	; no, but exit with a delay so as SPLASH is visible
; switchpress accepted!
	movlw	d'40' 		; wait another 400mS
	call	DELAYx10
	btfss	SW_PIN		; still being held on?
 	goto	EQ_DONE_DELAY	; NO, exit, do nothing

; if held longer >250mS , show that EQ mode is now enabled (until switch is released)
	bcf	MMI_FLAG,7	; clear switch flag
	bsf	CHG_FLAG,7	; & enable EQ mode (to run just ONCE!)
	call	CLEAR_LCD	; & clr SPLASH display

	call	TOPLINE
	movlw	0xD0		; offset for EQualise menu heading
	call	STROUT		; display it

EQ_DONE_DELAY
	movlw	d'50'		; ...for ~100x10mS = 1.0S so user is aware...
	call	DELAYx10	
EQ_TEST_DONE
; --------------------------------
; set WDT only after initialising LCD (as LCD takes a LOOONG time)
	banksel	WDTCON
	movlw	b'00010110'	; S/W WDT (31.5khz INTRC clk/65556) = 2.09S timeout
	movwf	WDTCON		; WDT prescaler (WDT enabled in config word)
	bsf	WDTCON,0	; SW enable WDT	
; ------------------------------------------------
ALLOW_INTERRUPTS
	banksel	PIE1		; BANK1
	bsf	PIE1,TMR1IE	; timer 1 overflow interrupt enable

	banksel	PIR1		; BANK0
	bcf	PIR1,TMR1IF	; timer 1 interrupt flag
	bsf	INTCON,PEIE	; enable periperal interrupts 	 
	bsf	INTCON,GIE	; enable global interrupts

; ********************************************************************************
MAIN_LOOP
; ********************************************************************************


CHG_MODE_SELECT:	; initial a/d read & charge mode selection
; ***************************************************
	clear_PWM		; in case entered via a charge mode, stop charging!
	call	READ_ALL_ADC
	errorflag_chk
; battery >13.2v?
	movfw	BATT_V_8BIT	; get 8 bit battery value
	sublw	FLT_ENTRY	; w = LITERAL - w
	btfss	CARRYBIT
	goto	FLOAT_STATE	; CARRY CLEAR, so battery > threshold
; battery >12.45?
	movfw	BATT_V_8BIT	; get 8 bit battery value
	sublw	BLK_ENTRY	; w = LITERAL - w
	btfss	CARRYBIT
	goto	BULK_STATE	; CARRY CLEAR, so battery > threshold
; battery >10.5?
	movfw	BATT_V_8BIT	; get w (=8 bit battery value)
	sublw	TKL_ENTRY	; L(thrshhld) - w = w
	btfss	CARRYBIT	; CARRY clear if w(battery) > L(threshold)
	goto	TRICKLE_STATE	; YES,CARRY CLEAR, so battery > threshold
; battery >9.5?
	movfw	BATT_V_8BIT	; get w (=8 bit battery value)
	sublw	BST_ENTRY	; L(thrshhld) - w = w
	btfss	CARRYBIT	; CARRY clear if w(battery) > L(threshold)
	goto	BURST_STATE	; YES,CARRY CLEAR, so battery > threshold
	goto	FAULT_STATE	; NO, CARRY SET, so battery > threshold


; ===============STATE.MACHINE.START===============
BURST_STATE:	; Applies short/strong bursts to revive battery if <10.5V.
; ===============STATE.MACHINE.START===============
	movlw	b'01000001'	; enable default display on entry
	movf	MMI_FLAG
	clrf	TCOMP_VAL	; not required when battery so low
	LOAD_OFF		; not allowed when battery so low
; --------------------------------
BURST_LOOP:
	movlw	0x10
	movwf	TABLE_OFFSET	; preload with required text-string offset
	call	MMI_UPDATE	; check switch and heartbeat flag, update LCD if needed
	addlw	0x00		; [force flags] if LCD was just updated then w=1, else w=0
	btfsc	ZEROBIT
	goto	BURST_LOOP	; w=0, just loop until LCD updates
; ---------  checks ---------
	call	READ_ALL_ADC	; w=1 (LCD updated) so read ADC
	errorflag_chk
	battlow_chk BV_MIN	; if below [parameter], goto CHG_MODE_SELECT, else continue
	duskflag_chk		; if prolonged no-sun, goto DUSK_STATE, else continue

; apply burst pulses of current till voltage >10.5V
	movf	CNTR_HBEAT,w	; INT updates this counter, Period = 262.144mS x 230 = 60.3Sec
	andlw	B'00001111'	; and when count = xxxx0000, turn PWM on for 1 in 16 = 6.25% duty cycle
 	btfss	ZEROBIT		;  = (16times x 0.262counter) = 4.2Sec every 60.3Sec
	goto	BURST_OFF
BURST_ON:
	movlw	d'63'		; set PWM for max charge
	banksel	CCPR1L		; Bank5
	movwf	CCPR1L
	banksel	d'0'
	goto	BURST_CHK	
BURST_OFF:
	clear_PWM		; stop charge
BURST_CHK:
; battery >10.5V?
	call	READ_BATT_V	; read A/D port
	movf	BATT_V_8BIT,w	; get w (=8 bit battery value)
	sublw	TKL_ENTRY	; L(thrshhld) - w = w
	btfsc	CARRYBIT	; CARRY clear if w(battery) > L(threshold)
	goto	BURST_LOOP	; CARRY SET, so battery < threshold
; --------------------------------
BURST_EXIT:			; fallthru to TRICKLE_STATE
; =================================================


; ===============STATE.MACHINE.START===============
TRICKLE_STATE:
; ===============STATE.MACHINE.START===============
	movlw	b'01000001'	; enable default display on entry
	movf	MMI_FLAG
	clrf	TCOMP_VAL	; not required when battery so low
	LOAD_OFF		; not allowed when battery so low	
; enable modest trickle charge rate
	movlw	D'10'	; about 15% of max
	banksel	CCPR1L		; Bank5
	movwf	CCPR1L
	banksel	d'0'
; --------------------------------
TRICKLE_LOOP:
	movlw	0x20
	movwf	TABLE_OFFSET	; preload with required text-string offset
	call	MMI_UPDATE	; check switch and heartbeat flag, update LCD if needed
	addlw	0x00		; [force flags] if LCD was just updated then w=1, else w=0
	btfsc	ZEROBIT
	goto	TRICKLE_LOOP	; w=0, just loop until LCD updates
; ---------  checks ---------
	call	READ_ALL_ADC	; w=1 (LCD updated) so read ADC
	errorflag_chk		; if any error flags set, goto FAULT_STATE, else continue
	battlow_chk TKL_FAIL	; if below [parameter], goto CHG_MODE_SELECT, else continue
	duskflag_chk		; if prolonged no-sun, goto DUSK_STATE, else continue

; is battery charged to >12.45V?
	movf	BATT_V_8BIT,w	; get 8 bit battery value
	sublw	BLK_ENTRY	; w = LITERAL - w
	btfsc	CARRYBIT	; CARRY clear if battery (w) > threshold
	goto	TRICKLE_LOOP
; --------------------------------
TRICKLE_EXIT:			; fallthru to BULK_STATE
; =================================================


; ===============STATE.MACHINE.START===============
BULK_STATE:	; charge at [constant] max current,  but should always be <C/8.
; ===============STATE.MACHINE.START===============
	movlw	b'01000001'	; enable default display on entry
	movf	MMI_FLAG
;	LOAD_ON
; set charge max to 13.8V (uncompensated)
	movlw	BLK_EXIT_H
	movwf	THRESH_HI 
	movlw	BLK_EXIT_L
	movwf	THRESH_LO
; init timers to see if BattV exit threshold reached in <60Secs (if so, we bypass EQ)
	clrf	CNTR_HBEAT	; clr counter
	clrf	CNTR_MINS	; clr counter
; can set max bulk charge rate here! Max current should be ~c/4, for 12AH = 3A
 	movlw	D'30'		; set a moderate start, about 50% of max
	banksel	CCPR1L		; Bank5
	movwf	CCPR1L
	banksel	d'0'
; --------------------------------
BULK_LOOP:
	movlw	0x30
	movwf	TABLE_OFFSET	; preload with required text-string offset
	call	MMI_UPDATE	; check switch and heartbeat flag, update LCD if needed
	addlw	0x00		; [force flags]  if LCD was just updated else w=0
	btfsc	ZEROBIT
	goto	BULK_LOOP	; w=0, just loop until LCD updates
; ---------  checks ---------
	call	READ_ALL_ADC	; w=1 (LCD updated) so read ADC
	call	TEMP_COMPENSATE	; temperature compensate Battery thresholds
	errorflag_chk		; if any error flags set, goto FAULT_STATE, else continue
	battlow_chk BLK_FAIL	; if below [parameter], goto CHG_MODE_SELECT, else continue
	duskflag_chk		; if prolonged no-sun, goto DUSK_STATE, else continue

	call	MAX_PWR		; (MAX PWR = MAX CHARGE CURRENT)

; Is battery charged >exit threshold?  [Compare X(THRSHLD H/L) with Y(BATT_V H/L)  
	compare_16 THRESH_HI_ADJ,THRESH_LO_ADJ,BATT_V_HI,BATT_V_LO
	goto	BULK_EXIT	; Y(BATT) => X(THRESH)	
	goto	BULK_LOOP	; Y(BATT) <  X(THRESH)
; --------------------------------
BULK_EXIT:	; where next? If BULK exits under 60Sec, then goto FLOAT 
	movf	CNTR_MINS,w
	btfss	ZEROBIT		; if Zero set, then <60S, so ABSORB disallowed
	goto	ABSORB_EQ_STATE	; Not Zero! so ABSORB (or EQUALISE) is still allowed
; BULK completed too quick, so battery fully charged, so disable ABS+EQ & goto FLOAT
	bcf	CHG_FLAG,6	; can't re-run ABS mode again today
	bcf	CHG_FLAG,7	; can't re-run EQ mode again today
	goto	FLOAT_STATE
; =================================================


; ===============STATE.MACHINE.START===============
ABSORB_EQ_STATE:  ; ABSORB & EQUALISE only follows BULK once-per-day (if flags allow)
; ===============STATE.MACHINE.START===============
	movlw	b'01000001'	; enable default display on entry
	movf	MMI_FLAG
;	LOAD_ON
; init timers, time allowed to be in this mode is limited
	clrf	CNTR_MINS	; clr counter
	clrf	CNTR_HRS	; clr counter

 	movlw	D'16'		; PWM moderate start, about 25% of max
	banksel	CCPR1L		; Bank5
	movwf	CCPR1L
	banksel	d'0'
; --------------------------------
ABS_EQ_LOOP
	btfss	CHG_FLAG,7	; Is EQmode allowed? (manually set)
	goto	TRY_ABSORB	; NO. this mode already run today

EQU_MODE	; set charge to (13.8V + 10%) for 1 hour ONCE PER YEAR (& not on GELL CELLS!)
	movlw	EQL_HOLD_H		
	movwf	THRESH_HI
	movlw	EQL_HOLD_L
	movwf	THRESH_LO
	movlw	0x50
	movwf	TABLE_OFFSET	; preload with required text-string offset
	
	goto	ABS_EQ_COMMON
; --------------------------------
TRY_ABSORB 
	btfss	CHG_FLAG,6	; Is ABSmode allowed?
	goto	ABS_EQ_EXIT	; NO. this mode already run today.
; ABS_MODE OK: set charge to 13.8V for 1 hour
	movlw	ABS_HOLD_H
	movwf	THRESH_HI
	movlw	ABS_HOLD_L
	movwf	THRESH_LO
	movlw	0x40
	movwf	TABLE_OFFSET	; preload with required text-string offset
; --------------------------------
ABS_EQ_COMMON
; update MMI when time to do so
	call	MMI_UPDATE	; check switch and heartbeat flag, update LCD if needed
	addlw	0x00		; [force flags] if LCD was just updated then w=1, else w=0
	btfsc	ZEROBIT
	goto	ABS_EQ_LOOP	; w=0, just loop until LCD updates
; ---------  checks ---------
	call	READ_ALL_ADC	; w=1 (LCD updated) so read ADC
	call	TEMP_COMPENSATE	; temperature compensate Battery thresholds
	errorflag_chk		; if any error flags set, goto FAULT_STATE, else continue
	battlow_chk ABS_FAIL	; if below [parameter], goto CHG_MODE_SELECT, else continue
	duskflag_chk		; if prolonged no-sun, goto DUSK_STATE, else continue

; increase PWM only if Vbatt < threshold [Compare X(THRSHLD H/L) with Y(BATT_V H/L)
	compare_16 THRESH_HI_ADJ,THRESH_LO_ADJ,BATT_V_HI,BATT_V_LO
	goto	ABS_EQ_DEC 	; Y(BATT) => X(THRESH)
	call	PWM_INCREASE
	goto	ABS_EQ_TIMER

ABS_EQ_DEC:
	call	PWM_DECREASE	; reduce charge voltage

ABS_EQ_TIMER:		; if timer >1Hr, then ABS_EQU mode completed
	btfss	CNTR_MINS,5	; >32Min keep it short for now
;	btfss	CNTR_HRS,0	; >60Min
	goto	ABS_EQ_LOOP	; loop until timeout
; --------------------------------
ABS_EQ_EXIT:			; fallthru to FLOAT_STATE, but disable ABS+EQ from re-running first...
	bcf	CHG_FLAG,6	; prevent any re-run of ABS mode today
	bcf	CHG_FLAG,7	; prevent any re-run of EQ  mode today
; --------------------------------


; ===============STATE.MACHINE.START===============
FLOAT_STATE:
; ===============STATE.MACHINE.START===============
	movlw	b'01000001'	; enable default display on entry
	movf	MMI_FLAG
;	LOAD_ON
; init counters
 	movlw	D'4'		; PWM low start
	banksel	CCPR1L		; Bank5
	movwf	CCPR1L
	banksel	d'0'
; so maintain FLOAT at 13v5 forever
	movlw	FLT_HOLD_H
	movwf	THRESH_HI
	movlw	FLT_HOLD_L
	movwf	THRESH_LO
; --------------------------------
FLOAT_LOOP
; update MMI when time to do so
	movlw	0x60
	movwf	TABLE_OFFSET	; preload with required text-string offset
	call	MMI_UPDATE	; check switch and heartbeat flag, update LCD if needed
	addlw	0x00		; [force flags] if LCD was just updated then w=1, else w=0
	btfsc	ZEROBIT
	goto	FLOAT_LOOP	; w=0, just loop until LCD updates
; ---------  checks ---------
	call	READ_ALL_ADC	; w=1 (LCD updated) so read ADC
	call	TEMP_COMPENSATE	; temperature compensate Battery thresholds
	errorflag_chk		; if any error flags set, goto FAULT_STATE, else continue
	battlow_chk FLT_FAIL	; if below [parameter], goto CHG_MODE_SELECT, else continue
	duskflag_chk		; if prolonged no-sun, goto DUSK_STATE, else continue

; increase PWM only if Vbatt < threshold [Compare X(THRSHLD H/L) with Y(BATT_V H/L) 
	compare_16 THRESH_HI_ADJ,THRESH_LO_ADJ,BATT_V_HI,BATT_V_LO
	goto	FLOAT_DEC 	; Y(BATT) => X(THRESH)	
;old	call	MAX_PWR		; Y(BATT) <  X(THRESH)
	call	PWM_INCREASE
	goto	FLOAT_LOOP
FLOAT_DEC:
	call	PWM_DECREASE	; reduce charge voltage
	goto	FLOAT_LOOP	; loop forever unless a fault!
; --------------------------------


; ===============STATE.MACHINE.START===============
DUSK_STATE:
; This state is entered (immediately) if Sun has gone, or (after ~30minutes) if Sun is insufficient for charging.
; During initial DUSK period, LOAD FET is ON (unless manually switched OFF) as long as BattV >12.5V
; After DUSK period (default 1Hr, max 4.25Hrs - or 256Hrs!), PIC is put into a true low-power SLEEP mode
; In SLEEP, PIC wakes every ~2 Seconds to look for return of Solar Volts or a long Switch press.
; ===============STATE.MACHINE.START===============
	clear_PWM		; stop PWM
	clrf	CNTR_DUSK	;
	LOAD_ON			; load auto-on for lights at night
; --------------------------------

DUSK_LOOP
	call	SCAN_SWITCH	; check switch, medium press toggles LOAD, long press will REBOOT
; time for LCD update?
	btfss	MMI_FLAG,6
	goto	DUSK_LOOP	; No
; --------------------------------
; Yes! Do a screen refresh
	bcf	MMI_FLAG,6	; clear MMI HBeat flag to prevent duplicate LCD refreshes
DUSK_MENU1:
	call	TOPLINE
	movlw	0x80		; offset for top line heading
	call	STROUT

	movlw	0x8c		; set to topline, position 13
	call	LCD_CMD

;-------------------------------------
; toggle a char ~2Sec to show PIC is alive
	btfss	MMI_FLAG,4
	goto	CHAR_ON
	bcf	MMI_FLAG,4
	movlw	' '
	goto	CHAROFF
CHAR_ON	bsf	MMI_FLAG,4
	movlw	':'
CHAROFF	call	LCD_CHR
;-----------------------------------

; calculate & display remaining time in DUSK mode
	movf	CNTR_DUSK,w	; exceeded maximum time allowed in DUSK mode? (in minutes)
	sublw	DUSK_MAX	; w = LITERAL - w threshold
	btfss	CARRYBIT	; CARRY clear if battery (w) > threshold
	goto	ASLEEP		; Yes, goto ASLEEP

; w reg still holds remaining minutes before SLEEP.counter to show Decimal minutes
	movwf	AARGB3 
	clrf	AARGB0
	clrf	AARGB1
	clrf	AARGB2
; convert AARGB2 & AARGB3 to BCD & ASCII for lcd display
	binary_2_bcd
	movf	ETWO,w		; display result as 3 decimal digits
	call	LCD_CHR
	movf	EONE,w
	call	LCD_CHR
	movf	EZERO,w
	call	LCD_CHR

	call	BOTTOMLINE
	call	SHOW_CELCIUS	; show Temperature is the default
	call	SPACE1
	movlw	'B' 
	call	LCD_CHR
	movlw	'a' 
	call	LCD_CHR
 	movlw	't' 
 	call	LCD_CHR
 	movlw	't' 
 	call	LCD_CHR
 	movlw	'=' 
 	call	LCD_CHR

	call	SHOW_BATT_V
	movlw	'V' 
	call	LCD_CHR
; --------------------------------

DUSK_MORE1
; auto-LOAD_OFF? (battery <12.5)?
	call	READ_BATT_V
	movfw	BATT_V_8BIT	; get 8 bit battery value
	sublw	BLK_ENTRY	; w = LITERAL - w 
	btfsc	CARRYBIT
	LOAD_OFF		; Yes, LOAD OFF but wait for timer to expire (in case SUN returns or SW pressed)

; is SUN fully back and OK for charging (ie no SUN error flags?)
	call	READ_SOLAR_V	; read..
	movf	ERR_FLAG,w	; get sun ERROR flags
	andlw	b'00000011'	; mask off all but ERR_FLAG  NoSUN<1>, LowSUN<0>
	btfsc	ZEROBIT	
	goto	DUSK_EXIT	; Yes, Solar V is good, but not a new day, so just restart
	goto	DUSK_LOOP	; No, loop in DUSK

DUSK_EXIT:	;SUN back but too soon for a new day, so just restart
	clrf	CHG_FLAG	; clears Sleepmode<0>, and note that no EQ<7> or ABS<6> allowed 
;	call	INIT_DISPLAY	; re-init LCD
	goto	MAIN_LOOP	; not a new day. RESTART, not REBOOT

;*******************************************
ASLEEP:
; For low power SLEEP mode, DISPLAY is TURNED OFF and PIC waits for sunrise or a switch press
; On exit from SLEEP (via full Sun or switch press) PIC does a full REBOOT, which reallows ABS (& EQ option) charging.
	movlw	b'11110011'	; set all PORTB high (except PWM & SWitch pins)	
	movwf	PORTB		; ...prior to removing ground from LCD.
	OPTO_OFF		; OPTO OFF (ie LCD, Temp sensor & Batt v.divider), current saved = 3.3mA
; turn LOADs & CLKs off
	LOAD_OFF		; (backlite 12.6mA saved) + LOAD current! 
	bcf 	T2CON,TMR2ON	; stop PWM source clock
;	bsf	OPTION_REG,7	; (WPU's off 0.23mA saved) BUT WERENT ON?
; --------------------------------
ASLEEP_LOOP
; disable ADC & sleep
	banksel	ADCON0
	bcf	ADCON0,ADON	; (ADC off 0.05mA saved)
	banksel	BEGIN

	SLEEP			; sleep for ~2.09Sec at total current (incl PCB) of 0.08mA 

; zzzzzzzz, but WDT will force resumption of program here every 2.09Sec
; turn on ADC and check for return of SOLAR_V
	banksel	ADCON0
	bsf	ADCON0,ADON	; A/D converter enabled
	banksel	BEGIN

	call	DELAY1mS	; let ADC stabilise, so chk SWITCH first...
; is button pressed?
	call	SCAN_SWITCH	; a long press will REBOOT, else nothing
; is SUN fully back (ie ok to charge?)
	call	READ_SOLAR_V	; read..
	movf	ERR_FLAG,w	; check Sun ERROR flags
	andlw	b'00000011'	; mask off all but ERR_FLAG  NoSUN<1>, LowSUN<0>
	btfss	ZEROBIT	
	goto	ASLEEP_LOOP	; still no sun.

ASLEEP_END:
; sun is back, wake-up with a FULL REBOOT
	bcf	INTCON,GIE	; global interrupts OFF
	bcf	WDTCON,0	; SW WDT OFF
	goto 	SETUP
; =================================================


; ===============STATE.MACHINE.START===============
FAULT_STATE:
; ===============STATE.MACHINE.START===============
	clrf	CNTR_HBEAT	; reset HB counter so display updates immediately
	clrf	CHG_FLAG	; EQallow<7>, ABSallow<6>, unused<5:1>, Sleep<0>
	clear_PWM		; clr PWM to stop charging
; display state
	call	TOPLINE
	movlw	0x00		; get topline string
	call	STROUT		; display it
; --------------------------------
ERR_LOOP:
	call	TOPLINE		; add ERR_FLAG (as hex) to topline
	movlw	0x86		; set  position 6
 	call	LCD_CMD
	movf	ERR_FLAG,w	
	call	BIN2HEX
; read switch, read AD sensors
	call	SCAN_SWITCH	; long press will reboot
	call	READ_ALL_ADC	; re-read all sensors and set/clear flags
; then sequentially check & display active errors
; ERR_FLAG: unused<7>, noTemp<6>, >40C<5>, <0C<4>, >15V<3>, <9V<2>, NoSUN<1>, LowSUN<0>
	btfsc	ERR_FLAG,2	; error = battery <9V.
	goto	ERR_BATT_LO
	btfsc	ERR_FLAG,3	; error = battery >15V
	goto	ERR_BATT_HI
BATTHILO_DONE:
	btfsc	ERR_FLAG,6	; error = TEMP sensor fail. Do before other temp check!
	goto	ERR_TEMPFAIL
	btfsc	ERR_FLAG,4	; error = TEMP <0C
	goto	ERR_TMPVAL_LO
	btfsc	ERR_FLAG,5	; error = TEMP >40C
	goto	ERR_TMPVAL_HI
TEMPHILO_DONE:
; retest fault flags, all clear yet?

; ---------  checks ---------
; --------------------------------
; auto-LOAD_OFF? (battery <12.5)?
	movf	ERR_FLAG,w	; error_FLAG check
	andlw	b'01111100'	; check for all except unused<7>,noSun<1>,sunLO<0>
	btfss	ZEROBIT
	goto	ERR_LOOP	; loop if any flag set, else exit
; no fault detected, so exit to MAIN loop
	clrf	MMI_FLAG	; clear any MMI flags
	bsf	MMI_FLAG,0	; & set LCD display back to default
	goto	MAIN_LOOP	; exit

;+++++++++++++++++++++++++++++++++
ERR_BATT_LO	; error = battery <9V
;+++++++++++++++++++++++++++++++++
	LOAD_OFF		; continue topline of LCD
	call	TOPLINE
	movlw	0x8a		; set to topline, position 11
	call	LCD_CMD
	movlw	'B'
	call	LCD_CHR
	movlw	'a'
	call	LCD_CHR
	movlw	't'
	call	LCD_CHR
	movlw	'='
	call	LCD_CHR
	movlw	'L'
	call	LCD_CHR
	movlw	'O'
	call	LCD_CHR

; --------------------------------
ERR_BATT_COMMON:
	call	BOTTOMLINE	; do bottomline of LCD
	show_load_state
; show Batt voltage for ~1Sec
	movlw	'B'
	call	LCD_CHR
	movlw	'='
	call	LCD_CHR
	call	SHOW_BATT_V
	movlw	d'100'
	call	DELAYx10
	goto	BATTHILO_DONE

;+++++++++++++++++++++++++++++++++
ERR_BATT_HI	; error = battery >15V
;+++++++++++++++++++++++++++++++++
	LOAD_OFF		; continue topline of LCD
	call	TOPLINE
	movlw	0x8a		; set to topline, position 11
	call	LCD_CMD
	movlw	'B'
	call	LCD_CHR
	movlw	'a'
	call	LCD_CHR
	movlw	't'
	call	LCD_CHR
	movlw	'='
	call	LCD_CHR
	movlw	'H'
	call	LCD_CHR
	movlw	'I'
	call	LCD_CHR
	goto	ERR_BATT_COMMON

;+++++++++++++++++++++++++++++++++
ERR_TMPVAL_LO			; error = TEMP <0C
;+++++++++++++++++++++++++++++++++
	call	TOPLINE		; continue topline of LCD
	movlw	0x8a		; set to topline, position 11
	call	LCD_CMD
	movlw	'T'
	call	LCD_CHR
	movlw	'm'
	call	LCD_CHR
	movlw	'p'
	call	LCD_CHR
	movlw	'='
	call	LCD_CHR
	movlw	'L'
	call	LCD_CHR
	movlw	'O'
	call	LCD_CHR
; --------------------------------
ERR_TEMP_BOTTOM:
; do bottomline of LCD
	call	BOTTOMLINE
	call	SHOW_CELCIUS
	call	SPACE4
; show Batt voltage
	movlw	'B'
	call	LCD_CHR
	movlw	'='
	call	LCD_CHR
	call	SHOW_BATT_V
ERR_TEMP_DELAY:
; display error for ~1Sec
	movlw	d'100'
	call	DELAYx10
	goto	TEMPHILO_DONE

;+++++++++++++++++++++++++++++++++
ERR_TMPVAL_HI			; error = TEMP >40C
;+++++++++++++++++++++++++++++++++
	call	TOPLINE		; continue topline of LCD
	movlw	0x8a		; set to topline, position 11
	call	LCD_CMD
	movlw	'T'
	call	LCD_CHR
	movlw	'm'
	call	LCD_CHR
	movlw	'p'
	call	LCD_CHR
	movlw	'='
	call	LCD_CHR
	movlw	'H'
	call	LCD_CHR
	movlw	'I'
	call	LCD_CHR
	goto	ERR_TEMP_BOTTOM

;+++++++++++++++++++++++++++++++++
ERR_TEMPFAIL	; error = TEMP sensor fail
;+++++++++++++++++++++++++++++++++
	call	TOPLINE		; continue topline of LCD	
	movlw	0x8a		; set to topline, position 11
	call	LCD_CMD
	movlw	'T'
	call	LCD_CHR
	movlw	'e'
	call	LCD_CHR
	movlw	'm'
	call	LCD_CHR
	movlw	'p'
	call	LCD_CHR
	movlw	'='
	call	LCD_CHR
	movlw	'?'
	call	LCD_CHR
; do bottomline of LCD
	call	BOTTOMLINE
	movlw	'A'
	call	LCD_CHR
	movlw	'/'
	call	LCD_CHR
	movlw	'D'
	call	LCD_CHR
	movlw	' '
	call	LCD_CHR
	movlw	'v'
	call	LCD_CHR
	movlw	'a'
	call	LCD_CHR
	movlw	'l'
	call	LCD_CHR
	movlw	'u'
	call	LCD_CHR
	movlw	'e'
	call	LCD_CHR
	CALL	SPACE1
; show temp sensor reading directly so fault can be analysed
 	movf	TEMP_V_HI,w
	call	BIN2HEX		; display error byte as hex
	movf	TEMP_V_LO,w
	call	BIN2HEX		; display error byte as hex
	goto	ERR_TEMP_DELAY
; ********************************************************************************




; ********************************************************************************
; CHARGING & PWM CONTROL *** CHARGING & PWM CONTROL *** CHARGING & PWM CONTROL *** 
; ********************************************************************************

; *************************************************
PWM_DECREASE:
; decrease PWM by difference between (excessive) Batt_V and current charging Threshold_V
; 		If Vdiff = 1 step  (35mV), then PWM decremented by 1
;		If Vdiff = 2 steps (70mV), then PWM decremented by 2
;		If Vdiff = 3 steps (105mV), then PWM decremented by 3
; 		If Vdiff > 3 steps (>105mV),then PWM stopped
; *************************************************
; first, preserve charge Threshold values prior to using subtract_16 macro
	movf	BATT_V_HI,w	
	movwf	TMPVAL_HI
	movf	BATT_V_LO,w
	movwf	TMPVAL_LO
; Get difference! X-Y=X (X is replaced with result, Y is preserved), and w is left holding X_Hi result.
	subtract_16	TMPVAL_HI,TMPVAL_LO, THRESH_HI_ADJ, THRESH_LO_ADJ
; HI byte of result ignored [should be 0, ie >4.4V diff].
	CLRCARRY		; divide LO byte result by 2 for 9bit resolution = 35mV/step
	rrf	TMPVAL_LO,f	; shift LO byte right, result in W, bit0 in C (keep)

; if battery >3steps (3x35mV = 105mV) above thrshld, switch off charge
 	movf	TMPVAL_LO,w	; load w with DIFF
 	sublw	D'3'		; L - w (3-DIFF) = w
 	btfss	CARRYBIT	; skip if DIFF 105mV or less
 	goto	DEC_CLR	; over 105mV, stop PWM

; otherwise decrease PWM by DIFF or switch off PWM
	movf	TMPVAL_LO,w	; move DIFF to w
	banksel	CCPR1L		; Bank5
	subwf	CCPR1L,w	; f(PWM) - w(DIFF) = w(proposed new PWM)
	btfss	CARRYBIT	; if resultant PWM <0, then set PWM to 0 (flatbottom)
	clrw			; result <0, so flatbottom (ie w=0, stop PWM)	
	write_PWM		; result >=0, so set PWM to w(proposed new PWM)

	goto	DEC_EXIT	; done
DEC_CLR:
	clear_PWM		; stop PWM
DEC_EXIT:
	return



; *************************************************
PWM_INCREASE:
; Battery_V is BELOW target Threshold_V, so find difference and increase PWM duty cycle by that difference!
; *************************************************

; first, preserve charge Threshold values prior to using subtract_16 macro
	movf	THRESH_HI_ADJ,w
	movwf	TMPVAL_HI
	movf	THRESH_LO_ADJ,w
	movwf	TMPVAL_LO
; X-Y=X (X is replaced with result, Y is preserved), and w is left holding X_Hi result.
	subtract_16	TMPVAL_HI, TMPVAL_LO, BATT_V_HI, BATT_V_LO
	movf	TMPVAL_LO,w
	sublw	d'3'		; L(x) - w (diff) = w . If diff >x, then CARRY CLR
	btfss	CARRYBIT
	goto	REDUCE_DIFF	; carry clr, so diff is >8
	goto	INC_DIFF_OK

REDUCE_DIFF	; compress diff above ''knee point' to minimise sudden V swings
; first, if net battery charging >0.5A, then dont increase PWM at all despite V difference.
; else, if difference is >3 then slowly increase pwm, 1 step at a time (35mV at a time)

; ++++++++++++++ CHARGE-CURRENT LIMITING ADDITION ++++++++++++++ 
PWM_INC_CHG_LIMIT:
; Batt chg current (=SolarI - LoadI) (NOTE: charge current should not exceed C/4)
	call	READ_SOLAR_I
	call	READ_BATT_I	; get current LOAD current (don't bother to check for LOAD_ON)
	movf	BATT_I_8BIT,w	; subtract LOAD-I (if present) from Solar-I = Batt-I 
	subwf	SOLAR_I_8BIT,w	; f - w = w
	btfss	CARRYBIT	; If Solar-I is so low that net result is <0, CARRY will go CLEAR!
	goto	PWM_CONTINUE	; CARRY is SET, result negative, no net Batt charge, so continue		
; battery charge current > c/4 limit?
	sublw	d'13'		; L(thrshhld) - w(Batt-I still in wReg) = w
	btfsc	CARRYBIT	; CARRY CLEAR if w(Batt-I) > L(0.5A threshold) ; CARRY Set if 
 	goto	PWM_CONTINUE	; CARRY is SET, Batt-I under limit, so just continue MPPT scan...
; use previous PWM value by decrementing current PWM value
	goto	INC_EXIT	; DO NOTHING, EXIT. Dec code will decrease PWM if need be.

PWM_CONTINUE:
  	movlw	d'1'		; slowly inc pwm
 	movwf	TMPVAL_LO	; store proposed PWM diff
 	movlw	d'33'		; allow time to stabilise V reading before acting
 	call	DELAYx10
INC_DIFF_OK
	movf	TMPVAL_LO,w	; carry set - small difference, use as-is, so reload diff.
	banksel	CCPR1L		; Bank5
	addwf	CCPR1L,w	; w(diff) + PWM = w(proposed PWM)
	banksel	d'0' 
 	movwf	TMPVAL_LO	; store proposed PWM value
; check if proposed PWM value needs to be flattopped
	sublw	D'63'		; Over 63? L(PWM limit 63) - w(new PWM) = w
	btfss	CARRYBIT	; CARRY only SET if result was less than 63
	goto	INC_FLATTOP	; CARRY CLR, overlimit, so flattop at 63

	movf	TMPVAL_LO,w	; CARRY SET, all good, so reload proposed PWM value
	write_PWM		; load it to PWM

INC_EXIT
	call	DELAY10mS	; allow V change to settle before reassesing
	call	DELAY10mS
	return
INC_FLATTOP
	movlw	D'63'		; flattop at maximum 63
	goto	INC_EXIT



; ********************************************************************************
; MPPT scan. 8bit version that works ok!!!
; ********************************************************************************

; *************************************************
MAX_PWR:
; Get max power by periodically sweeping PWM duty cycle from 0 to 64, then using the CCPR1L value that had MAX power
; INT cycles CNTR_HBEAT every 60Sec, when zero INT sets MPPT_FLAG<6> to indicate a scan is due
; this routine acks with MPPT_FLAG<7> = scan in progress, and only SCAN_END clears both flags.
; * new addition:  if BATT_CHG_LIMIT current (SolarI - LoadI) ever >2.5A (c/4), then PWM value is halved and SCAN_ENDED .
; *************************************************
MPPT_BEGIN
; is a PWM scan already in progress?
	btfsc	MPPT_FLAG,7
	goto	INC_CHRG	; Yes, continue a run
; or time for a new PWM scan?
	btfss	MPPT_FLAG,6	; flag set by INT every 67 secs
	goto	MPPT_EXIT	; NO. just exit		
MPPT_INIT			; No. initialise a run
	bcf	MPPT_FLAG,6	; clear initiator
	bsf	MPPT_FLAG,7	; indicate a SCAN has started 
	clrf	PWR_HI		; clr power
	clrf	PWR_LO
  	clrf	CCPR1L_STORE
	clear_PWM		; clr PWM
	call	DELAY10mS	; allow time for current reading to drop
; ------------------------
INC_CHRG			; ramp thru full PWM duty cycle
	read_PWM		; has PWM reached maximum duty cycle?
	sublw	D'63'
	btfss	CARRYBIT	; CARRY SET if w below threshold
	goto	SCAN_DONE	; CARRY CLR, so terminate
	inc_PWM			; CARRY SET, so inc PWM and continue MPPT run
; ------------------------
; CALC_SOLAR_PWR: multiply 8-bit SOLAR_V by SOLAR_I (gives relative power, not WATTS)
 	movlw	d'2'		; allow time to stabilise at I and V inputs
 	call	DELAYx10	
	call	READ_SOLAR_V
; ++++++++++++++ CHARGE-CURRENT LIMITING ADDITION ++++++++++++++ 
BATT_CHG_LIMIT:
; Batt chg current (=SolarI - LoadI) (NOTE: charge current should not exceed C/4)
	call	READ_SOLAR_I
	call	READ_BATT_I	; get current LOAD current (don't bother to check for LOAD_ON)
	movf	BATT_I_8BIT,w	; subtract LOAD-I (if present) from Solar-I = Batt-I 
	subwf	SOLAR_I_8BIT,w	; f - w = w
	btfss	CARRYBIT	; If Solar-I is so low that net result is <0, CARRY will go CLEAR!
	goto	MPPT_CONTINUE	; CARRY is SET, result negative, no net Batt charge, so continue		
; battery charge current > c/4 limit?
	sublw	CHG_I_LIMIT	; L(thrshhld) - w(Batt-I still in wReg) = w
	btfsc	CARRYBIT	; CARRY CLEAR if w(Batt-I) > L(C/8 threshold) ; CARRY Set if 
 	goto	MPPT_CONTINUE	; CARRY is SET, Batt-I under limit, so just continue MPPT scan...
; use previous PWM value from current scan (it was just under limit) by decrementing current PWM value
	dec_PWM			; CARRY is CLEAR, Batt-I over limit, so decrement... 
	movwf	CCPR1L_STORE	; ...save PWM...
	goto	SCAN_DONE	; ...then terminate MPPT scan with PWM set at Battery-I limit.
MPPT_CONTINUE:
; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	movf	SOLAR_V_8BIT,w
	movwf	AARGB0
	movf	SOLAR_I_8BIT,w
	movwf	BARGB0
	multiply_8bit		; input AARGB0 x BARGB0 = output AARGB0,AARGB1
; Compare X with Y. is PWR a new maximum?
	compare_16 AARGB0,AARGB1,PWR_HI,PWR_LO
	goto	SCAN_MORE	; Y(previous power)is => X(new power)
				; X(new power) > Y(old power). So save PWM!
SAVE_MAX
; new power is larger than last, so update power values with this new maximum
	movf	AARGB0,w
	movwf	PWR_HI
	movf	AARGB1,w
	movwf	PWR_LO
	banksel	CCPR1L		; Bank5
	movf	CCPR1L,w
	banksel	d'0'
	movwf	CCPR1L_STORE	; & save current PWM value

SCAN_MORE
	call	DELAY10mS	; allow time to stabilise at I and V inputs
 	goto	INC_CHRG	; loop, do full scan in one hit

SCAN_DONE
; PWM has completed testing its full range 
	clrf	MPPT_FLAG	; clear all flags, scan complete
	movf	CCPR1L_STORE,w	; get PWM that provided best power
	write_PWM		; and use it to set charge rate at maximum power
	movlw	0x10		; stabilise 100mS before returning
	call	DELAYx10

MPPT_EXIT
	return


; ********************************************************************************
; 			*** I/O SUBROUTINES *** I
; ********************************************************************************

; ***************************************************
SCAN_SWITCH:
; looks for a switch press, and if so, cycles MMI_FLAG & updates LCD
; Exits with MMI_FLAG updated & MMI_FLAG,7 set to indicate new keypress
; ***************************************************
	bcf	MMI_FLAG,7	; clear last keypress
	btfss	SW_PIN		; switch pressed?
	goto	SCAN_EXIT	; no, just exit

; a switch was pressed, if <10mS treat it as a glitch
	movlw	d'1'		; wait 10mS
	call	DELAYx10	; Debounce line
	btfss	SW_PIN		; still pressed?
	goto	SCAN_EXIT	; no, so exit

; switchpress accepted! if released <300mS then rotate menus 
WAIT_FOR_RELEASE
	movlw	d'30' 
	call	DELAYx10	; wait 300mS more
	btfss	SW_PIN		; released yet?
 	goto	SW_OFF_SHORT	; YES, short press only, so rotate menus

; if released >300mS but <600mS then toggle LCD backlite & LOAD
	movlw	d'30' 
	call	DELAYx10	; wait 300mS more
	btfss	SW_PIN		; released yet?
	goto	SW_OFF_LONG	; YES but longer press, so toggle backlite

; if released before 1.5Sec then do nothing, otherwise REBOOT!
	movlw	d'90' 
	call	DELAYx10	; wait another ~900mS
	btfss	SW_PIN		; released yet?
	goto	SCAN_EXIT	; YES. exit without doing anything at all!

SW_REBOOT:			; no. switch held too long! reboot.
	bcf	INTCON,GIE	; disable global interrupts
	bcf	WDTCON,0	; SW disable WDT for the moment!	
	goto	SETUP		; then reboot!

; yes, so toggle backlite using LOAD ON/OFF FET.
SW_OFF_LONG
	btfss	PORTA,6		; is backlite ON?
	goto	BL_ON		; no
	LOAD_OFF		; yes, so switch it off & exit
	goto	SCAN_EXIT	; exit without indicating switch was pressed
BL_ON	LOAD_ON			; turn backlite ON
	goto	SCAN_EXIT	; exit without indicating switch was pressed

SW_OFF_SHORT			
; cycle menu pointer
	clrc			; do this before the rlf!
	rlf	MMI_FLAG,f	; update display type
	btfss	MMI_FLAG,4	; rolled too far?
	goto	CYCLE_DONE	; no...
	clrf	MMI_FLAG	; ...yes
	bsf	MMI_FLAG,0	; so go back to first menu
CYCLE_DONE
	bsf	MMI_FLAG,7	; indicate new keypress, done AFTER menu flag RLF!

SCAN_EXIT			; common exit for pass or fail
	return

; ***************************************************
MMI_UPDATE:
	; updates ONLY if interrupt sets [MMI_FLAG,6] or NEW SWITCH press [MMI_FLAG,7]
	; updates display differently according to current MENU flag
	; Enter with TABLE_OFFSET,w preloaded to match current charge mode string
; ***************************************************
	call	SCAN_SWITCH
 	btfss	MMI_FLAG,7	; button just pressed?
 	goto	LCD_HBEAT	; NO. so check for heartbeat

	bcf	MMI_FLAG,7	; YES. clear flag & update LCD now
	goto	LCD_REFRESH
LCD_HBEAT			
	btfss	MMI_FLAG,6	; time to update MMI (only cleared by LCD update)
	retlw	d'0'		; not time, so just exit with FAIL flag (0)

LCD_REFRESH
	bcf	MMI_FLAG,6	; clear MMI HBeat flag

	btfsc	MMI_FLAG,0	; show Default display?
	goto	MENU1
	btfsc	MMI_FLAG,1	; show detailed Solar display?
	goto	MENU2
	btfsc	MMI_FLAG,2	; show detailed Battery display?
	goto	MENU3
	btfsc	MMI_FLAG,3	; show detailed Load display?
	goto	MENU4
	goto	MENU_EXIT	; exit (but codec should never fall thru to here!)

; --------------------------------
MENU1		; current charge mode display
; --------------------------------
	call	TOPLINE
; --------------------------------
; is NoSUN display required?
MENU1_T_NS	
	btfss	ERR_FLAG,1	; is NoSUN flag set?
	goto	MENU1_T_LS	; no
	movlw	0x80		; show NoSUN string
	call	STROUT
	goto	MENU1_T_ALT
; is LowSUN display required?
MENU1_T_LS	
	btfss	ERR_FLAG,0	; is LowSUN flag set?
	goto	MENU1_T		; no, so do default topline
	movlw	0x70		; show LowSUN string
	call	STROUT

; --------------------------------
MENU1_T_ALT
	movlw	0x89		; set to topline, position 9
	call	LCD_CMD
	movlw	'M' 
	call	LCD_CHR
	movlw	'i' 
	call	LCD_CHR
 	movlw	'n' 
 	call	LCD_CHR
 	movlw	's' 
 	call	LCD_CHR
; --------------------------------
; toggle a char ~2Sec to show PIC is alive
	btfss	MMI_FLAG,4
	goto	CHR_ON
	bcf	MMI_FLAG,4
	movlw	' '
	goto	CHR_OFF
CHR_ON	bsf	MMI_FLAG,4
	movlw	':'
CHR_OFF	call	LCD_CHR
; --------------------------------
; get counter to show minutes
	movf	CNTR_DUSK,w
	movwf	AARGB3 
	clrf	AARGB0
	clrf	AARGB1
	clrf	AARGB2
; convert AARGB2 & AARGB3 to BCD & ASCII for lcd display
	binary_2_bcd
	movf	EONE,w		; display NO SUN minutes count-up
	call	LCD_CHR
	movf	EZERO,w
	call	LCD_CHR
	goto	MENU1_B		; NO SUN topline done, goto bottomline
; --------------------------------
MENU1_T
	movf	TABLE_OFFSET,w	; [offset is preloaded by current charge mode]
	call	STROUT
	movlw	0x8a		; set to topline, position 10
	call	LCD_CMD
	call	SHOW_BATT_V

MENU1_B	
	call	BOTTOMLINE
; --------------------------------
; show Equalise (if flag set), else Celcius display
	btfss	CHG_FLAG,7	; EQualise flag set?
	goto	MENU1_B1	; No, just show temperature as normal 
	movlw	'E'		; Yes, so indicate EQ mode ON to warn user
	call	LCD_CHR
	movlw	'Q'		;
	call	LCD_CHR
	movlw	'!'		; 
	call	LCD_CHR
	goto	MENU1_B2
; --------------------------------
MENU1_B1	
	call	SHOW_CELCIUS	; show Temperature is the default
MENU1_B2
	movlw	0xc3		; set to bottomline, position 4 (as temp. display width could vary)
	call	LCD_CMD
	call	SPACE1

	movlw	's'		; prefix to indicate SolarV
	call	LCD_CHR
	call	SHOW_SOLAR_V	; (already just read A/D for PWR calculation)
	call	SPACE1		; add spaces
	movlw	'p'	
	call	LCD_CHR
	movlw	'w'	
	call	LCD_CHR
	movlw	'm'	
	call	LCD_CHR

	read_PWM		; show PWM as Decimal

	movwf	AARGB3 
	clrf	AARGB0
	clrf	AARGB1
	clrf	AARGB2
	binary_2_bcd		; convert AARGB3 to BCD & ASCII
	movf	EONE,w		; display pwm as ASCII
	call	LCD_CHR
	movf	EZERO,w
	call	LCD_CHR

	goto	MENU_EXIT


; --------------------------------
MENU2	; solar display
; --------------------------------
	call	TOPLINE
	movlw	0xA0		; offset for top line heading
	call	STROUT
	movlw	0x89		; set to topline, position 8
	call	LCD_CMD
	call	SPACE1		; add spaces
 	call	SHOW_SOLAR_PWR

	call	BOTTOMLINE
 	call	SHOW_CELCIUS

	movlw	0xc4		; set to bottomline, position 5 (as temp. display width can vary)
	call	LCD_CMD
	call	SPACE1		; add space

	call	SHOW_SOLAR_V	; (already just read A/D for PWR calculation)
	call	SPACE1		; add spaces
	call	SHOW_SOLAR_I	; (already just read A/D for PWR calculation)
	goto	MENU_EXIT

; --------------------------------
MENU3	; battery display
; --------------------------------
	call	TOPLINE
	movlw	0xB0		; offset for top line heading
	call	STROUT
	movlw	0x89		; set to topline, position 11
	call	LCD_CMD
	call	SPACE1		; add space
	call	SHOW_BATT_V

	call	BOTTOMLINE
	call	SHOW_CELCIUS

	movlw	0xc4		; set to bottomline, position 5 (as temp. display width can vary)
	call	LCD_CMD
	call	SPACE1		; add space

	movlw	'T'
 	call	LCD_CHR
	movlw	'c'
 	call	LCD_CHR
	movlw	'm'
 	call	LCD_CHR
	movlw	'p'
 	call	LCD_CHR
	movlw	'='
 	call	LCD_CHR
	call	SHOW_TEMPCOMP
	goto	MENU_EXIT

; --------------------------------
MENU4	; LOAD display
; --------------------------------
	call	TOPLINE
	show_load_state
	call	SPACE1		; add spaces
	movlw	'='
 	call	LCD_CHR
	movlw	' '
 	call	LCD_CHR
	call	SHOW_BATT_I

	call	BOTTOMLINE
	call	SHOW_CELCIUS

	movlw	0xc4		; set to bottomline, position 5 (as temp. display width can vary)
	call	LCD_CMD
	call	SPACE1		; add space

	movlw	'B'
 	call	LCD_CHR
	movlw	'a'
 	call	LCD_CHR
	movlw	't'
 	call	LCD_CHR
	movlw	't'
 	call	LCD_CHR
	movlw	'='
	call	LCD_CHR
	call	SHOW_BATT_V

; --------------------------------
MENU_EXIT
; --------------------------------
	retlw	d'1'		; return with SUCCESS flag (1)



; ********************************************************************************
; 		 *** LCD ROUTINES ***  
; ********************************************************************************


; ***************************************
STROUT:		; send STRing OUT to LCD
; ***************************************
; This code only good for extended mid-range 16bit INDF/FSR calls for lookup
; The table can be placed ANYWHERE on PAGE1,2,3 or 4! 

; An ORG declaration & STR_TABLE: label must be placed at start of strings (16 x 16 max = 256bytes)
; Enter with W = OFFSET from TABLE_BASE (the start of the required 16chr string block)
; NOTE each string must start on a $xx00 boundary and be 16chrs long (or be terminated by $00) 

	IF (high (STR_TABLE) != (high (STR_TABLE_END)-1))
		ERROR "KB table hits page boundary!"
		ENDIF

	movwf	FSR1L		; save LO byte for table read (string_start offset)
	movlw	HIGH STR_TABLE	; get HI byte of table start address
	movwf	FSR1H		; save as HI byte for table read
STROUT_LOOP
	movf	INDF1,w		; get a character
	btfsc	ZEROBIT		; is it a string terminator($00)?
	goto	STROUT_END	; yes, so exit early 
	call	LCD_CHR		; else display it

	incf	FSR1L		; inc string pntr
	movf	FSR1L,w		; get updated string ptr
	andlw	b'00001111'	; chk LO nibble, has 16 chr limit rolled over?
	btfss	ZEROBIT		; if ZERO, then all done, exit
	goto	STROUT_LOOP	; else loop & get next character
	
STROUT_END
	return	



; ***************************************
INIT_DISPLAY:	; init display in 4-byte mode
; ***************************************
; LCD connected to Hi-nibble of PortB; Lo-nibble of LCD is grounded. PortB0= /EN, PortB1= /RS.
; LCD defaults to 8-bit mode, must be initialised into 4-bit mode & data must then be sent as nibbles.

; replaced bit manipulation in this code (BSF, BCF) in case it had potential for read/modify/write error


; Clear LCD /ENable and RegisterSelect (/RS=command mode, RS=data mode)
	movf	PORTB,w		; read LCD port
	andlw	b'11111100'	; set /EN_PIN & RS_PIN low (rB1 & rB0)
	movwf	PORTB		; write back to port
; wait for busy to become clear 
	movlw	d'2'		; LCD is BUSY for 10mS after initial power-on
	call	DELAYx10	; so put in 20mS safety delay

; write FUNCTION_SET $30
	movf	PORTB,w		; read LCD port
	iorlw	b'00110000'	; make Hi-nibble = $3
	movwf	PORTB		; write to PortB
	call	PULSE_EN	; toggle /EN Hi to Lo (write to LCD)
	movlw	d'50'		; wait min. 4.1mS - so make it 5mS(8MHz clk)
	movwf	TIME1		; (TIME1 is a x100uS multiplier)
	call	DLY_outer
; write FUNCTION_SET $30 for 2nd time
	call	PULSE_EN	; toggle /EN Hi to Lo (write to LCD)
	call	DELAY100uS	; min 40uS delay this time - so make it 100uS (8MHz clk)
; write FUNCTION_SET $30 for 3rd time
	call	PULSE_EN	; ; toggle /EN Hi to Lo (write to LCD)
	call	DELAY100uS	; min 40uS delay this time - so make it 100uS (8MHz clk)

; write FUNCTION_SET $20 = set to 4-bit mode (using Hi-nibble only, and while keeping /RS low)
	movf	PORTB,w		; read LCD port
	andlw	b'11101101'	; 1st FUNCTION_SET $20 [clear D4_PIN & RS_PIN (rB4 & rB1)]
	movwf	PORTB		; write back to port
	call	PULSE_EN	; toggle /EN Hi to Lo (write to LCD)
	call	DELAY100uS	; min 40uS delay - so make it 100uS (8MHz clk)
; write FUNCTION_SET $24 again, this time as full 8-bits (sent as Hi/Lo nibbles)
	movlw	b'00101000'	; 2nd FUNCTION_SET $20, this time setting 2-line mode & 5x10 font
	call	LCD_CMD		; write full 8-bits, sent as Hi/Lo nibbles

	movlw	b'00001000'	; DISPLAY_OFF
	call	LCD_CMD
	call	CLEAR_LCD	; DISPLAY_CLR
	movlw	B'00000110'	; CHAR ENTRY MODE (cursor moves right, display not shifted) / h'06' ; set entry mode (increm addr, no shift)
	call	LCD_CMD	
	movlw	b'00001100'	; DISPLAY_ON ( + cursor & blinking off) 
	call	LCD_CMD

	return


; ***************************************	
; Send LCD Data (CMD or CHR) as 2 nibbles	
; ***************************************	
LCD_CMD:	; enter & exits with W = LCD COMMAND data
	bcf	RS_PIN		; RS = 0 = LCD command mode
	goto	SND_NIBBLES
LCD_CHR:	; enter & exits with W = LCD ASCII characer
	bsf	RS_PIN		; RS = 1 = LCD character mode

SND_NIBBLES
; upper nibble
	movwf	LCD_DATA	; store W
	movf	PORTB,w		; read portB to W,  W = (xxxx+yyyy)
	andlw	0x0F		; clear upper bits, W = (0000+yyyy)
	movwf	PORTB		; now Port B =(0000+yyyy)
	movf	LCD_DATA,w	; W = char(msb+lsb)
	andlw	0xF0		; W = char(msb+0000)	
	iorwf	PORTB,f		; Port B =(msb+yyyy)
	call	PULSE_EN
; lower nibble
	movf	PORTB,w		; read portB to W,  W = (xxxx+yyyy)
	andlw	0x0F		; clear upper bits, W = (0000+yyyy)
	movwf	PORTB		; now Port B =(0000+yyyy)
	swapf	LCD_DATA,w	; swap nibbles, result in W = char(lsb+msb)
	andlw	0xF0		; W = char(lsb+0000)
	iorwf	PORTB,f		; Port B =(lsb+yyyy)
PULSE_EN
	bsf	EN_PIN		; pulse /ENable high for ~2uS
	nop
	nop			; (added 2 more NOPs for 8MHz)
	nop	
	bcf	EN_PIN		; set /EN low again
BUSYWAIT
	movlw	D'140'		; [+1] No R/W, so cant read BUSY status
	movwf	LCD_BSY		; [+1] so wait 200uS (min) instead. 
BUSY_1 	decfsz	LCD_BSY,f	; [+1] 
	goto	BUSY_1		; [+2] =(2+(140x3)) @ 0.5uS = 210uS (@8MHz clk)

	movf	LCD_DATA,w	; restore W
	return


; ***************************************	
; LCD manipulation subroutines	
; ***************************************
CLEAR_LCD:
	movlw	B'00000001'	; clear LCD command
	call	LCD_CMD
	call	DELAY1mS	; new LCD clears in 100uS so make it 1mS to be safe.
	return
TOPLINE:
	movlw	0x80		; cursor address  = top line far left
 	call	LCD_CMD
	return
BOTTOMLINE:
	movlw	0xC0		; cursor address  = bottom line far left
 	call	LCD_CMD
	return

SPACE5:
	movlw	0x20		; space
	call	LCD_CHR
SPACE4:
	movlw	0x20		; space
	call	LCD_CHR
SPACE3:
	movlw	0x20		; space
	call	LCD_CHR
SPACE2:
	movlw	0x20		; space
	call	LCD_CHR
SPACE1:
	movlw	0x20		; space
	call	LCD_CHR
	return

LCD_LINE1_W:
	addlw	0x80		; move to 1st row, column = W
	call	LCD_CMD
	return
LCD_LINE2_W:
	addlw	0xC0		; move to 2nd row, column = W
	call	LCD_CMD
	return

SHOW_OFF:
	movlw	'O'
 	call	LCD_CHR
	movlw	'F'
 	call	LCD_CHR
	movlw	'F'
 	call	LCD_CHR
	return
SHOW_ON:
	movlw	'O'
 	call	LCD_CHR
	movlw	'N'
 	call	LCD_CHR
	movlw	' '
 	call	LCD_CHR
	return


; ***************************************
SPLASH:		; startup SPLASH display
; ***************************************
	call	TOPLINE
	movlw	0xE0		; offset for ninth string
	call	STROUT
	call	BOTTOMLINE
	movlw	0xF0		; offset for tenth string
	call	STROUT
	return


; ***************************************
BIN2HEX:	; display contents of W as "$xx"
; ***************************************
	movwf	BIN_INPUT	; TEMP save w
	movlw	A'$'
	call	LCD_CHR		; show '$' to indicate that HEX chr is coming
	
	call	BIN2HEX_CONV
	movwf	HEX1_OUT	; save & show MSB in ASCII 
	call	LCD_CHR	
	
	call	BIN2HEX_CONV
	movwf	HEX2_OUT	; save & show LSB in ASCII 
	call	LCD_CHR	
BIN2HEX_END	
	return
; --------------
BIN2HEX_CONV	
	swapf	BIN_INPUT,f	; swap
	movf	BIN_INPUT,w	; get value
	andlw	b'00001111'	; mask off high nibble
	movwf	HEX2_OUT	; TEMP save in currently unused register
	
	sublw	0x09		; literal-w = w. is w>$0A?
	btfsc	CARRYBIT	; Carry clear if w>literal, CARRY set w <= literal.
	goto	BIN2HEX1	; CARRY clear, Wresult > $20
	
	movlw	0x07
	addwf	HEX2_OUT,f	; yes ,greater than 9, add 7
BIN2HEX1
	movf	HEX2_OUT,w	; get W	again
	addlw	0x30		; make chr 0-9 or A-F
	return
; ***************************************




; ********************************************************************************
; 	*** GENERAL SUBROUTINES ***  
; ********************************************************************************

INIT_RESET:	; clear ram
	movlw	0x20		; start of RAM addressing
	movwf	FSR0L		; pointer
LOOP_CLEAR
	clrf	INDF0		; clear values
	incf	FSR0L,f
	btfss	FSR0L,7		; if bit 7 set finished (20 to 7F cleared)
	goto	LOOP_CLEAR
	return





; ***************************************************
; TIME DELAYS
; ***************************************************
DELAYx10:			; 10mS delay x Arg W	
	movwf	TIME_INIT	; multiplier
DLYX_1	call	DELAY10mS
	decfsz	TIME_INIT,f	; dec multiplier
	goto	DLYX_1		; loop x Arg W
	return

DELAY10mS:			
	movlw	d'100'		; nominal 10ms delay (8MHz clk)
	movwf	TIME1		; TIME1 = timeloop repeat value
	call	DLY_outer
	return

DELAY1mS:				
	movlw	d'10'		; nominal 1ms delay (8MHz clk)
	movwf	TIME1		; TIME1 = timeloop repeat value
	call	DLY_outer
	return

DELAY100uS:
	movlw	d'1'		; nominal 100uS delay (8MHz clk)
	movwf	TIME1		; TIME1 = timeloop repeat value
DLY_outer			; 2+ 100x100-1 +2return =20201cycles (20mS@4MHz, 10mS@8MHz)
	movlw	d'66'
	movwf	TIME2		; TIME2 is internal loop 100uS preset	
DLY_inner			; 4 + 3x64 + 4 = 200cycles (200uS@4MHz, 100uS@8MHz)
	decfsz	TIME2,f
	goto	DLY_inner	; loop inner
	decfsz	TIME1,f
	goto	DLY_outer	; loop outer
	return


; ********************************************************************************
; 	*** ADC READ & DISPLAY ROUTINES ***
; ********************************************************************************

; ***************************************************
READ_ALL_ADC:	; read all A/D channels
; ***************************************************
; All SOLAR readings must be done while still under load
	call	READ_SOLAR_V
	call	READ_SOLAR_I
; pause charging before reading Battery voltage othewise will read charge voltage!
	banksel	CCPR1L		; Bank5
	movf	CCPR1L,w
	banksel	d'0'
	movwf	CCPR1L_STORE	; save current PWM value

	banksel	CCPR1L		; Bank5
	clrf	CCPR1L
	banksel	d'0'

	movlw	d'2'
	call	DELAYx10	; then let battery voltage settle
	call	READ_BATT_I
	call	READ_TEMP_V
	call	READ_BATT_V

; restore charging 
	movf	CCPR1L_STORE,w
	write_PWM		; restore PWM value
	return


; ***************************************************
ADC_CONVERT:	; A/D acquisition
; ***************************************************
; Banksel not needed; all registers in ADCONx Bank1, and RAM variables common to every bank.
	bcf	INTCON,GIE	; stop interrupt  (REGISTER common to all Banks)

; Acquisition Delay: A/D AMP & cap must settle after channel selection (min 5uS, use 10uS)
	movlw	D'20'		; * 8MHz operation = (20 x 0.5uS = 10uS)
	movwf	ADC_WAIT	; wait ~10uS ADC sample acquisition time
SAMPLE	decfsz	ADC_WAIT,f
	goto	SAMPLE

;do a conversion
 	banksel	ADCON0
	bsf	ADCON0,ADGO	; start conversion (set GO/DONE bit)
ADC_DO	btfsc	ADCON0,ADGO	; conversion done when GO/DONE bit clear (~11 cycles)
	goto	ADC_DO

;reduce to 8 bits
	banksel	ADRESH
	movf	ADRESH,w	; ms byte
	movwf	TMPVAL_HI	; make a copy (RAM variable common to all Banks
	movf	ADRESL,w	; ls bits
	movwf	TMPVAL_LO	; make a copy (RAM variable common to all Banks)
	clrc			; (not really needed, as HI byte is discarded at the end!)
	 rrf	TMPVAL_HI,f	; shift HI byte right, result in F, bit0 in C (keep)
	 rrf	TMPVAL_LO,f	; shift LO byte right, C to bit7, bit0 in C (ignored)
	 rrf	TMPVAL_HI,f	; final rrf of HI byte, new bit0 in C (keep)
	 rrf	TMPVAL_LO,w	; shift LO again, C to bit7, new C (ignored), result in W
	movwf	TMPVAL_8BIT	; copy W  (RAM variable common to all Banks)

	bsf	INTCON,GIE	; re-allow interrupt (REGISTER common to all Banks)
	return


; ***************************************************
READ_BATT_V:	; AD0 - battery voltage range 0-17.76V
		; input is divided by 4.7/(4.7+12k) = 0.2814
; ***************************************************
	banksel	ADCON0
	bcf	ADCON0,2	;  set A/D to read CH0 (b'000')
	bcf	ADCON0,3
	bcf	ADCON0,4 
	call	ADC_CONVERT

	banksel	ADRESL
	movf	ADRESL,w	; ls bits
	banksel	d'0'		; Bank0
	movwf	BATT_V_LO	; battery voltage ls byte
	banksel	ADRESL
	movf	ADRESH,w	; ms byte
	banksel	d'0'		; Bank0
	movwf	BATT_V_HI	; battery voltage ms byte

; save the last 8bit value from ADC conversion
	movf	TMPVAL_8BIT,w	; get last A/D 8bit result
	movwf	BATT_V_8BIT	; copy it 

; ---------------------------------------
; check if over/under batt voltage limits & set fault flags only! 
; ----------------------------------------
CHK_BATT_HI;	battery >15v?
	movf	BATT_V_8BIT,w	; get 8 bit battery value
	sublw	BV_MAX		; w = L (15V) - w (BattV)
	btfss	CARRYBIT	; CARRY clear if w(BattV) > L(threshold)
	goto	BATTHI_FAULT	; CARRY is CLR, so >15v
	
CHK_BATT_LO:	; battery <9v (ie battery faulty)?
	movf	BATT_V_8BIT,w	; get 8 bit battery value
	sublw	BV_MIN		; w = LITERAL - w
	btfsc	CARRYBIT
	goto	BATTLO_FAULT	; CARRY SET, so <9V fault
; so all good!
	movlw	b'11110011'	; clear all battery volts fault flags
	andwf	ERR_FLAG,f
	goto	CHK_BATT_DONE

BATTHI_FAULT
; CARRY SET, so battery under 15V threshold
	 bsf	ERR_FLAG,3	; set >15V flag
	 bcf	ERR_FLAG,2	; clear <9V flag
	 goto	CHK_BATT_DONE	; flag st, so exit

BATTLO_FAULT	;<9V fault
	 bsf	ERR_FLAG,2	; set <9V flag
	 bcf	ERR_FLAG,3	; clear >15V flag
CHK_BATT_DONE
	return


; ***************************************************
READ_BATT_I:	; AD1 - battery load current, 0-5.0A range
		; 0.1ohm @ 5Amp = 500mV x gain 10 = 5V
; ***************************************************
	banksel	ADCON0
	bsf	ADCON0,2	;  set A/D to read CH1 (b'001')
	bcf	ADCON0,3
	bcf	ADCON0,4

	call	ADC_CONVERT

	banksel	ADRESL
	movf	ADRESL,w	; ls bits
	banksel	d'0'		; Bank0
	movwf	BATT_I_LO	; load current ls byte
	banksel	ADRESL
	movf	ADRESH,w	; ms byte
	banksel	d'0'		; Bank0
	movwf	BATT_I_HI	; load current ms byte

; save the last 8bit value from ADC conversion
	movf	TMPVAL_8BIT,w	; get last A/D 8bit result
	movwf	BATT_I_8BIT	; copy it 
	return


; ***************************************************
READ_SOLAR_V:	; AN2 - solar cell voltage 0-28.40V
		; input is divided by 4k7/(4k7+22k) = 0.176
; ***************************************************
	banksel	ADCON0
	bcf	ADCON0,2	;  set A/D to read CH2 (b'010')
	bsf	ADCON0,3
	bcf	ADCON0,4

	call	ADC_CONVERT

	banksel	ADRESL
	movf	ADRESL,w	; ls bits
	banksel	d'0'		; Bank0
	movwf	SOLAR_V_LO	; solar cell voltage ls byte
	banksel	ADRESL
	movf	ADRESH,w	; ms byte
	banksel	d'0'		; Bank0
	movwf	SOLAR_V_HI	; solar cell voltage ms byte

; save the last 8bit value from ADC conversion
	movf	TMPVAL_8BIT,w	; get last A/D 8bit result
	movwf	SOLAR_V_8BIT	; copy it

; ---------------------------------------
; Range check solar voltage, set & clear flags AND save/restore/stop PWM 
; ---------------------------------------
NOSUN_CHK			; is solar panel active?
	movf	SOLAR_V_8BIT,w	; get w = SolarV
	sublw	SV_FAIL		; w = LITERAL - w. (is solarV =<1V ?)
	btfss	CARRYBIT	; CARRY clear if w(solarV) > L(threshold)
	goto	SUNLOW_CHK	; CARRY is CLR, so panel is >1V

NOSUN				; CARRY SET, so panel <1V (its probably DUSK or NIGHT)
	clrf	MPPT_FLAG	; kill any in-progress MPPT run
	clear_PWM		; stop PWM
	bcf	ERR_FLAG,0	; set LOWSun flag and..
	bsf	ERR_FLAG,1	; set NOSun flag
	goto	SOLAR_DONE	; exit

;----------------------------
SUNLOW_CHK			; is solarV sufficient for charging (>12.5V)?
	movf	SOLAR_V_8BIT,w	; get w = SolarV
	sublw	SV_LOW		; w = LITERAL - w. (is solarV =<12.5V ?)
	btfss	CARRYBIT	; CARRY clear if w(solarV) > L(threshold)
	goto	SUN_OK		; CARRY is CLR, all good!

SUN_IS_LOW			; SolarV too low for charging		
	clrf	MPPT_FLAG	; kill any in-progress MPPT run
	clear_PWM		; stop PWM
	bsf	ERR_FLAG,0	; set LOWSun flag 
	bcf	ERR_FLAG,1	; clr NOSun flag
	goto	SOLAR_DONE	; exit
;----------------------------
SUN_OK	bcf	ERR_FLAG,0	; clr LowSUN flag
	bcf	ERR_FLAG,1	; clr NoSUN flag

SOLAR_DONE
	return


; ***************************************************
READ_SOLAR_I:	; AN3 -  solar cell current - 0-5.0A range
		; 0.1ohm @ 5Amp = 500mV x gain 10 = 5V
; ***************************************************
 banksel	ADCON0
	bsf	ADCON0,2	;  set A/D to read CH3 (b'011')
	bsf	ADCON0,3
	bcf	ADCON0,4 

	call	ADC_CONVERT

	banksel	ADRESL
	movf	ADRESL,w	; ls bits
	banksel	d'0'		; Bank0
	movwf	SOLAR_I_LO	; solar cell current ls byte
	banksel	ADRESL
	movf	ADRESH,w	; ms byte
	banksel	d'0'		; Bank0
	movwf	SOLAR_I_HI	; solar cell current ms byte

; save the last 8bit value from ADC conversion
	movf	TMPVAL_8BIT,w	; get last A/D 8bit result
	movwf	SOLAR_I_8BIT	; copy it 

	return


; ***************************************************
READ_TEMP_V:	; AD4 - temperature sensor
		; 10mV / degreeC kelvin (2.73V = 0C, 2.93V = 20C)
; ***************************************************
 	banksel	ADCON0
	bcf	ADCON0,2	; set A/D to read CH4 (b'100)
	bcf	ADCON0,3
	bsf	ADCON0,4

	call	ADC_CONVERT

	banksel	ADRESL
	movf	ADRESL,w	; ls bits
	banksel	d'0'		; Bank0
	movwf	TEMP_V_LO	; temperature ls byte
	banksel	ADRESL
	movf	ADRESH,w	; ms byte
	banksel	d'0'		; Bank0
	movwf	TEMP_V_HI	; temperature ms byte

; save the last 8bit value from ADC conversion
	movf	TMPVAL_8BIT,w	; get last A/D 8bit result
	movwf	TEMP_V_8BIT	; copy it

; ---------------------------------------
; Error check & set fault flags only! 
; ---------------------------------------
CHK_VERY_LO
; is HI byte = 0000001x? i.e. is temp >250K (-23C)?
	btfsc	TEMP_V_HI,1	; is Hi-byte bit-1 set?
	goto	CHK_VERY_HI	; yes. ok so far
	bsf	ERR_FLAG,6	; no. very cold! add sensor fail <6> flag
	goto	CHK_TEMP_DONE
CHK_VERY_HI
; is HI byte = 00000011? i.e. is temp >375K (102C)?
	btfss	TEMP_V_HI,0	; is Hi-byte bit-0 also set?
	goto	CHK_TEMP	; no. ok so far
	bsf	ERR_FLAG,6	; yes. very hot! add sensor fail <6> flag
	goto	CHK_TEMP_DONE

CHK_TEMP
; is temp. >0 Celcius?
	movfw	TEMP_V_LO ;,w	; get LO byte from temp.sensor
	sublw	TV_MIN		; L(thrshhld) - w = w
	btfsc	CARRYBIT	; CARRY clear if w(t.sensor) > L(threshold)
	goto	CHK_TOOCOLD	; CARRY set, so <0degrees C
; is temp. >40 Celcius? 
	movf	TEMP_V_LO,w	; get LO byte from temp.sensor
	sublw	TV_MAX		; L(thrshhld) - w = w
	btfss	CARRYBIT	; CARRY clear if w(t.sensor) > L(threshold)
	goto	CHK_TOOHOT	; CARRY clear, so >40degrees C
; sensor is reading between 0C and 40C, so all good!
	movlw	b'10001111'	; clear ALL temperature fault flags
	andwf	ERR_FLAG,f
	goto	CHK_TEMP_DONE 

CHK_TOOCOLD	; <0 celcius, set flags	& stop charging
	bsf	ERR_FLAG,4	; set <0C <4> flag
	goto	CHK_TEMP_DONE
CHK_TOOHOT	; >40 celcius, set flags & stop charging
	bsf	ERR_FLAG,5	; set >40C <5> flag

CHK_TEMP_DONE
	return


; ***************************************************
SHOW_BATT_V:
; ***************************************************
; load ADC values for multiplication
	clrf	AARGB0
	movf	BATT_V_HI,w	; ms of count
	movwf	AARGB1
	movf	BATT_V_LO,w	; ls byte
	movwf	AARGB2
;next adjust A/D count to (A/Dcount x 2)+1
	CLRCARRY		; prepare for shift
	rlf	AARGB1,f	; multiply msb x2, ADmax =$3ff so cant rollover
	CLRCARRY		; prepare for possible lsb rollover
	rlf	AARGB2,f	; multiply lsb x2
	btfsc	CARRYBIT	; result >$ff?
	incf	AARGB1,f	; yes, adjust msb
	incf	AARGB2,f	; add 1 to lsb (compensate for 0.5bit A/D error)
; no need to check for rollover as lsb is always 0 after doubling.

; then multiply FSD conversion factor:
	clrf	BARGB0
	movlw	BV_LO
	movwf	BARGB1
	clrf	BARGB2
	multiply_2424	; AARGB0/B1/B2 x BARGB0/B1/B2 => AARGB0/B1/B2/B3/B4/B5
	binary_2_bcd	; convert result (already in AARGB0/B1/B2/B3) to BCD and then to ASCII
; display it
	movf	ETHREE,w
 	call	LCD_CHR
	movf	ETWO,w
 	call	LCD_CHR
	movlw	'.'
	call	LCD_CHR
	movf	EONE,w
 	call	LCD_CHR
	movf	EZERO,w
 	call	LCD_CHR
	movlw	'V'
 	call	LCD_CHR
	return

; ***************************************************
SHOW_BATT_I:
; ***************************************************
; load ADC values for multiplication
	clrf	AARGB0
	movf	BATT_I_HI,w	; ms of count
	movwf	AARGB1
	movf	BATT_I_LO,w	; ls byte
	movwf	AARGB2

;next adjust A/D count to (A/Dcount x 2)+1
	CLRCARRY		; prepare for shift
	rlf	AARGB1,f	; multiply msb x2, ADmax =$3ff so cant rollover
	CLRCARRY		; prepare for possible lsb rollover
	rlf	AARGB2,f	; multiply lsb x2
	btfsc	CARRYBIT	; result >$ff?
	incf	AARGB1,f	; yes, adjust msb
	incf	AARGB2,f	; add 1 to lsb (compensate for 0.5bit A/D error)
; (no need to check for rollover as lsb is always 0 after doubling)
; then multiply FSD conversion factor:
	clrf	BARGB0
	movlw	BI_LO
	movwf	BARGB1
	clrf	BARGB2
	multiply_2424	; AARGB0/B1/B2 x BARGB0/B1/B2 => AARGB0/B1/B2/B3/B4/B5
	binary_2_bcd	; convert result (already in AARGB0/B1/B2/B3) to BCD and then to ASCI
; display it
	movf	ETHREE,w	; ensure msb is still "0"
	sublw	'0'		; msb rolled over from 0x.xA to 1x.xA?
	btfss	CARRYBIT
	goto	I_OVERLOAD

	movf	ETWO,w
	call	LCD_CHR
	movlw	'.'
	call	LCD_CHR
	movf	EONE,w
	call	LCD_CHR
	movf	EZERO,w
 	call	LCD_CHR
	movlw	'A'
 	call	LCD_CHR
	return

I_OVERLOAD:
	movlw	'o'
	call	LCD_CHR
	movlw	'v'
	call	LCD_CHR
	movlw	'e'
	call	LCD_CHR
	movlw	'r'
	call	LCD_CHR
	return

; ***************************************************
SHOW_SOLAR_V:
; ***************************************************
; load ADC values for multiplication
	clrf	AARGB0
	movf	SOLAR_V_HI,w	; ms of count
	movwf	AARGB1
	movf	SOLAR_V_LO,w	; ls byte
	movwf	AARGB2

;next adjust A/D count to (A/Dcount x 2)+1
	CLRCARRY		; prepare for shift
	rlf	AARGB1,f	; multiply msb x2, ADmax =$3ff so cant rollover
	CLRCARRY		; prepare for possible lsb rollover
	rlf	AARGB2,f	; multiply lsb x2
	btfsc	CARRYBIT	; result >$ff?
	incf	AARGB1,f	; yes, adjust msb
	incf	AARGB2,f	; add 1 to lsb (compensate for 0.5bit A/D error)
; no need to check for rollover as lsb is always 0 after doubling.

; then multiply FSD conversion factor:
	movlw	SV_HI		; conversion factor
	movwf	BARGB0
	movlw	SV_LO
	movwf	BARGB1
	clrf	BARGB2
	multiply_2424	; AARGB0/B1/B2 x BARGB0/B1/B2 => AARGB0/B1/B2/B3/B4/B5
	binary_2_bcd	; convert result (already in AARGB0/B1/B2/B3) to BCD and then to ASCI
; display it
	movf	ETHREE,w
 	call	LCD_CHR
	movf	ETWO,w
 	call	LCD_CHR
	movlw	'.'
	call	LCD_CHR
	movf	EONE,w
 	call	LCD_CHR
	movlw	'V'
 	call	LCD_CHR
	return

; ***************************************************
SHOW_SOLAR_I:
; ***************************************************
; load ADC values for multiplication
	clrf	AARGB0
	movf	SOLAR_I_HI,w	; ms of count
	movwf	AARGB1
	movf	SOLAR_I_LO,w	; ls byte
	movwf	AARGB2

;next adjust A/D count to (A/Dcount x 2)+1
	CLRCARRY		; prepare for shift
	rlf	AARGB1,f	; multiply msb x2, ADmax =$3ff so cant rollover
	CLRCARRY		; prepare for possible lsb rollover
	rlf	AARGB2,f	; multiply lsb x2
	btfsc	CARRYBIT	; result >$ff?
	incf	AARGB1,f	; yes, adjust msb
	incf	AARGB2,f	; add 1 to lsb (compensate for 0.5bit A/D error)
; no need to check for rollover as lsb is always 0 after doubling.
; then multiply FSD conversion factor:
	clrf	BARGB0
	movlw	SI_LO
	movwf	BARGB1
	clrf	BARGB2
	multiply_2424	; AARGB0/B1/B2 x BARGB0/B1/B2 => AARGB0/B1/B2/B3/B4/B5
	binary_2_bcd	; convert result (already in AARGB0/B1/B2/B3) to BCD and then to ASCI
; display it
	movf	ETHREE,w
	sublw	'0'		; msb rolled over from 0x.xA to 1x.xA?
	btfss	CARRYBIT
	goto	I_OVERLOAD
	movf	ETWO,w
	call	LCD_CHR
	movlw	'.'
	call	LCD_CHR
	movf	EONE,w		; only show 0.1A res
 	call	LCD_CHR
 	movf	EZERO,w
 	call	LCD_CHR
	movlw	'A'
 	call	LCD_CHR
	return

; ***************************************************
SHOW_SOLAR_PWR:
; ***************************************************
;  first, raw multiplication of V x I  (Note: ADC reading is NOT (doubled + 1) for POWER calculations)
	clrf	AARGB0
	movf	SOLAR_V_HI,w
	movwf	AARGB1
	movf	SOLAR_V_LO,w
	movwf	AARGB2

	clrf 	BARGB0
	movf	SOLAR_I_HI,w
	movwf	BARGB1
	movf	SOLAR_I_LO,w
	movwf	BARGB2

	multiply_2424	; AARGB0/B1/B2 x BARGB0/B1/B2 => AARGB0/B1/B2/B3/B4/B5
;  realign result...
	clrf	AARGB0
	movf	AARGB3,w
	movwf	AARGB1
	movf	AARGB4,w
	movwf	AARGB2
; then enter SOLAR POWER 284.0W FSD conversion argument
	movlw	SP_HI
	movwf	BARGB0
	movlw	SP_LO
	movwf	BARGB1
	clrf	BARGB2

	multiply_2424	; AARGB0/B1/B2 x BARGB0/B1/B2 => AARGB0/B1/B2/B3/B4/B5
	binary_2_bcd	; convert result (already in AARGB0/B1/B2/B3) to BCD and then to ASCI
; display it
	movf	EFOUR,w
 	call	LCD_CHR
	movf	ETHREE,w
 	call	LCD_CHR
	movf	ETWO,w
	call	LCD_CHR
	movlw	'.'
	call	LCD_CHR
	movf	EONE,w	
 	call	LCD_CHR
	movlw	'W'
 	call	LCD_CHR
	return

; ***************************************************
SHOW_CELCIUS:
; ***************************************************
; ADC read of temp sensor has already parsed value & set/clear any out-of-range charge flags
; preload registers for multiplication
	clrf	AARGB0
	movf	TEMP_V_HI,w	; ms of count
	movwf	AARGB1
	movf	TEMP_V_LO,w	; ls byte
	movwf	AARGB2

;next adjust A/D count to (A/Dcount x 2)+1
	CLRCARRY		; prepare for shift
	rlf	AARGB1,f	; multiply msb x2, ADmax =$3ff so cant rollover
	CLRCARRY		; prepare for possible lsb rollover
	rlf	AARGB2,f	; multiply lsb x2
	btfsc	CARRYBIT	; result >$ff?
	incf	AARGB1,f	; yes, adjust msb
	incf	AARGB2,f	; add 1 to lsb (compensate for 0.5bit A/D error)
; no need to check for rollover as lsb is always 0 after doubling.
; then multiply FSD conversion factor:
	movlw	KV_HI		; Temp Volts (Kelvin) conversion factor
	movwf	BARGB0
	movlw	KV_LO
	movwf	BARGB1
	clrf	BARGB2
	multiply_2424		; AARGB0/B1/B2 x BARGB0/B1/B2 => AARGB0/B1/B2/B3/B4/B5
; convert Kelvin result (ARG2 & ARG3) to Celcius by subtracting ADC of 273K(0deg) = ((558.6*2)+1) x $271(KVH/L conversion factor) = $AA9[7E]
	movlw	h'A9'		; divisor LSB ( not rounded up to AAA) 
	subwf	AARGB3,f	; subtract & resave new lsb
	btfsc	CARRYBIT
	goto	DO_MSB
	decf	AARGB2,f	; needed CARRY, so decrement MSB
DO_MSB
	movlw	h'0A'		; divisor MSB
	subwf	AARGB2,f	; subtract & resave new msb
	btfsc	CARRYBIT
	goto	OVER_ZERO

UNDER_ZERO
; CARRY needed, so result was below 0 Celcius
	movlw	'-'		; temp below 0C, so add negative sign	
	call	LCD_CHR
	comf	AARGB2,f	; invert result 
	comf	AARGB3,f
	incf	AARGB3,f	; I dont want a simple 2's complement, so add 1
	btfsc	ZEROBIT		; and at -16C & -32C etc, bump msb too.. 
	incf	AARGB2,f	; bump MSB

OVER_ZERO
; convert to BCD and then ASCII (NOTE: input already in AARGB0 to AARGB3) 
	binary_2_bcd
; display it
	movf	ETWO,w
	call	LCD_CHR
	movf	EONE,w
	call	LCD_CHR
;	movlw	'.'		; last digit not used
;	call	LCD_CHR
;	movf	EZERO,w
;	call	LCD_CHR
	movlw	h'df'		; degrees character (usually $dF or $b2)
	call	LCD_CHR
	movlw	'C'
	call	LCD_CHR
	return


; ***************************************************
TEMP_COMPENSATE: ; battery charge threshold temperature compensation
; ***************************************************
; enter with W loaded with LO byte of current ADC temperature reading
; enter with THRESH_HI/LO preloaded (uncompensated Voltage Threshold)
; exits with THRESH_HI/LO_ADJ values (temperature-compensated V-Threshold)
; 	adjustment is -25mV/C for temperatures >20C
; process:
; 	calculate the difference above 20deg C and Ambient (in #ADC units)
; 	convert this difference to #bits in BATT_V units (=17mV/bit) 
; 	no adjustment under 20C as battery life improvement neglible
; only used for BURST/ABSORB/EQ/FLOAT charge modes

	clrf	TCOMP_VAL	; erase last reading
	movf	THRESH_LO,w
	movwf	THRESH_LO_ADJ	; copy LO byte
	movf	THRESH_HI,w
	movwf	THRESH_HI_ADJ	; copy HI byte
; is temperature >20 celcius?
	movlw	KV20_LO		; (LO byte of ADcount at 20C trip level)
	subwf	TEMP_V_LO,w	; subtract LO byte of ADcount of ambient Temp 
	btfss	CARRYBIT	; CARRY SET if >20C
	goto	TEMPCOMP_EXIT	; no action at this stage, just exit

OVER_TWENTY
; Wreg still holds difference (in ADcount units) btwn ambient and 20C.
	movwf	AARGB2		; place W (value above 20C) into lsb.
	clrf	AARGB0
 	clrf	AARGB1
; multiply by conversion factor to make it #bits @ BATT_V resolution
	clrf	BARGB0
	movlw	TCA_HI		; Temp Comp Adjustment conversion factor
	movwf	BARGB1
	movlw	TCA_LO
	movwf	BARGB2
	multiply_2424
; subtract result from THRESHOLD
	movf	AARGB3,w	; get result = #bits @ BATT_V resolution
	movwf	TCOMP_VAL	; then save this value for later LCD display
	subwf	THRESH_LO,w	; subtract this value from THRSHLD (unaltered) with result in w
	movwf	THRESH_LO_ADJ	; save W as new adjusted threshold
	btfsc	CARRYBIT	; Is there rollover & need to decrement from HI byte? 	
	goto	TEMPCOMP_EXIT	; no
	decf	THRESH_HI_ADJ,f
TEMPCOMP_EXIT
	return


; ***************************************************
SHOW_TEMPCOMP:	; show temperature compensation in mV
; ***************************************************
; preload registers for multiplication
	clrf	AARGB0
	movf	TCOMP_VAL,w	; = #bits of temp compensation x 256
	movwf	AARGB1
	clrf	AARGB2
; multiply by conversion factor
	clrf	BARGB0
	movlw	TC_HI		; Temp.Comp. conversion factor
	movwf  	BARGB1
	movlw	TC_LO
	movwf	BARGB2
	multiply_2424
; then convert to BCD and ascii characters for lcd display
; NOTE: input already in AARGB2 & AARGB3, outputs BCD and ASCII! 
	binary_2_bcd
; display it
	movlw	'-'		; compensation is always MINUS!
	call	LCD_CHR
	movf	ETWO,w
	call	LCD_CHR
	movf	EONE,w
 	call	LCD_CHR
	movf	EZERO,w
 	call	LCD_CHR
	movlw	'm'
 	call	LCD_CHR
	movlw	'V'
 	call	LCD_CHR
	return

; ***************************************************







; ********************************************************************************
	org     0x800		; page 1 ($0800 to $0FFF)
; use macros to call these PAGE1 calls (use pagesel with EMR devices, else adjust PCLATH before & after)
; ********************************************************************************

; ********************************************************************************
; MATH ROUTINES - UNSIGNED FIXED POINT
; FXM2424U	24x24 Bit Unsigned Fixed Point Multiply 24x24 -> 48
; FXD3232U	32/32 Bit Unsigned Fixed Point Divide 32/32 -> 32.32
; EIGHTEIGHT	8 x 8 multiply

; ********************************************************************************
; Subroutine BCD (to convert 32-bit binary to 10 digit BCD)
; Input: binary in AARGB0(msb), AARGB1, AARGB2 & AARGB3(lsb)
; Result: BCD values in BCD1, BCD2, BCD3, BCD4 & BCD5.
;         BCD1 is overflow indicator, BCD2 is MSB, BCD5 is LSB
; also creates hex-ascii characters of same for display
; EEIGHT	; hex ascii of BCD (overflow)
; ESEVN & ESIX	; msb hexascii digit pair
; EFIVE & EFOUR
; ETHREE & ETWO
; EONE & EZERO	; lsb hexascii digit pair
; ********************************************************************************
BIN_BCD24
	clrf	AARGB2		; clear more significant bytes
BIN_BCD34
	clrf	AARGB1
	clrf	AARGB0
BIN_BCD:
BINBCDX
	bcf	STATUS,C 	; clear CARRY bit
	movlw	D'32'
	movwf	COUNT		; 32 in count
	clrf	BCD1		; set BCD registers to 0	
	clrf	BCD2
	clrf	BCD3
	clrf	BCD4
	clrf	BCD5

LOOPBCD	
	rlf	AARGB3,f	; LSB shift left binary registers
	rlf	AARGB2,f
	rlf	AARGB1,f
	rlf	AARGB0,f	; MSB
	rlf	BCD5,f		; LSB shift left BCD registers
	rlf	BCD4,f
	rlf	BCD3,f
	rlf	BCD2,f
	rlf	BCD1,f		; MSB

	decfsz	COUNT,f		; reduce count value return when 0
	goto	DECADJ		; continue decimal adjust
	
; result in BCD1-5. (BCD1 overrange, BCD2 MS byte)
	movf	BCD1,w
	andlw	0x0F
	iorlw	0x30
	movwf	EEIGHT

	swapf	BCD2,w		; get ms nibble
	andlw	0x0F
	iorlw	0x30		; convert to ASCII
	movwf	ESEVN		; ms digit
	movf	BCD2,w		; get 2nd ms nibble
	andlw	0x0F
	iorlw	0x30		; convert to ASCII
	movwf	ESIX

	swapf	BCD3,w		; get next nibble
	andlw	0x0F
	iorlw	0x30		; convert to ASCII
	movwf	EFIVE		; ms digit
	movf	BCD3,w		; get next nibble
	andlw	0x0F
	iorlw	0x30		; convert to ASCII
	movwf	EFOUR
	
	swapf	BCD4,w		; get ms nibble
	andlw	0x0F
	iorlw	0x30		; convert to ASCII
	movwf	ETHREE		; ms digit
	movf	BCD4,w		; get 2nd ms nibble
	andlw	0x0F
	iorlw	0x30		; convert to ASCII
	movwf	ETWO

	swapf	BCD5,w		; get ms	nibble
	andlw	0x0F
	iorlw	0x30		; convert to ASCII
	movwf	EONE		; ms digit
	movf	BCD5,w		; get 2nd ms nibble
	andlw	0x0F
	iorlw	0x30		; convert to ASCII
	movwf	EZERO
	return			; completed decimal to BCD operation

; subroutine decimal adjust
DECADJ	
	movlw	BCD5		; BCD LSB address
	movwf	FSR0L		; pointer for BCD5
	call	ADJBCD		; subroutine to adjust BCD
	movlw	BCD4
	movwf	FSR0L
	call	ADJBCD
	movlw	BCD3
	movwf	FSR0L
	call	ADJBCD
	movlw	BCD2
	movwf	FSR0L
	call	ADJBCD
	movlw	BCD1
	movwf	FSR0L
	call	ADJBCD
	goto	LOOPBCD

; subroutine adjust BCD
ADJBCD	
	movlw	0x03		; w has 03	
	addwf	INDF0,w		; add 03 to BCDx register (x is 1-5)
	movwf	TEMP		; store w
	btfsc	TEMP,3		; test if >7
	movwf	INDF0		; save as LS digit
	movlw	0x30		; 3 for MSbyte
	addwf	INDF0,w		; add 30	to BCDx register
	movwf	TEMP		; store w
	btfsc	TEMP,7		; test if >7
	movwf	INDF0		; save as MS digit
	return			; end subroutine


; ********************************************************************************
; 24x24 Bit Unsigned Fixed Point Multiply 24x24 -> 48
; Input: 24 bit unsigned fixed point multiplicand in AARGB0,1,2
; 24 bit unsigned fixed point multiplier in BARGB0,1,2
; Use: CALL FXM2424U
; Output: 48 bit unsigned fixed point product in AARGB0
; Result: AARG <-- AARG x BARG
; Max Timing: 9+501+2 = 512 clks
; Min Timing: 9+150 = 159 clks
; ********************************************************************************
FXM2424U:			
	clrf	AARGB3	; clear partial product
	clrf	AARGB4	
	clrf	AARGB5	
	movf	AARGB0,w	
	movwf	TEMPB0	
	movf	AARGB1,w	
	movwf	TEMPB1	
	movf	AARGB2,w	
	movwf	TEMPB2	
	movlw	H'08'
	movwf	LOOPCOUNT	
LOOPUM2424A			
	rrf	BARGB2,f	
	btfsc	STATUS,C	
	goto	ALUM2424NAP	
	decfsz	LOOPCOUNT,f	
	goto	LOOPUM2424A	
	movwf	LOOPCOUNT	
LOOPUM2424B	
	rrf	BARGB1,f	
	btfsc	STATUS,C	
	goto	BLUM2424NAP	
	decfsz	LOOPCOUNT,f	
	goto	LOOPUM2424B	
	movwf	LOOPCOUNT	
LOOPUM2424C	
	rrf	BARGB0,f	
	btfsc	STATUS,C	
	goto	CLUM2424NAP	
	decfsz	LOOPCOUNT,f	
	goto	LOOPUM2424C	
	clrf	AARGB0	
	clrf	AARGB1	
	clrf	AARGB2	
	retlw	0x00	
CLUM2424NAP	
	bcf	STATUS,C	
	goto	CLUM2424NA	
BLUM2424NAP	
	bcf	STATUS,C	
	goto	BLUM2424NA	
ALUM2424NAP	
	bcf	STATUS,C	
	goto	ALUM2424NA	
ALOOPUM2424	
	rrf	BARGB2,f	
	btfss	STATUS,C	
	goto	ALUM2424NA	
	movf	TEMPB2,w	
	addwf	AARGB2,f	
	movf	TEMPB1,w	
	btfsc	STATUS,C	
	incfsz	TEMPB1,w	
	addwf	AARGB1,f	
	movf	TEMPB0,w	
	btfsc	STATUS,C	
	incfsz	TEMPB0,w	
	addwf	AARGB0,f	
ALUM2424NA	
	rrf	AARGB0,f	
	rrf	AARGB1,f	
	rrf	AARGB2,f	
	rrf	AARGB3,f	
	decfsz	LOOPCOUNT,f	
	goto	ALOOPUM2424	
	movlw	H'08'
	movwf	LOOPCOUNT	
BLOOPUM2424	
	rrf	BARGB1,f	
	btfss	STATUS,C	
	goto	BLUM2424NA	
	movf	TEMPB2,w	
	addwf	AARGB2,f	
	movf	TEMPB1,w	
	btfsc	STATUS,C	
	incfsz	TEMPB1,w	
	addwf	AARGB1,f	
	movf	TEMPB0,w	
	btfsc	STATUS,C	
	incfsz	TEMPB0,w	
	addwf	AARGB0,f	
BLUM2424NA	
	rrf	AARGB0,f	
	rrf	AARGB1,f	
	rrf	AARGB2,f	
	rrf	AARGB3,f	
	rrf	AARGB4,f	
	decfsz	LOOPCOUNT,f	
	goto	BLOOPUM2424	
	movlw	H'08'
	movwf	LOOPCOUNT	
CLOOPUM2424
	rrf	BARGB0,f	
	btfss	STATUS,C	
	goto	CLUM2424NA	
	movf	TEMPB2,w	
	addwf	AARGB2,f	
	movf	TEMPB1,w	
	btfsc	STATUS,C	
	incfsz	TEMPB1,w	
	addwf	AARGB1,f	
	movf	TEMPB0,w	
	btfsc	STATUS,C	
	incfsz	TEMPB0,w	
	addwf	AARGB0,f	
CLUM2424NA
	rrf	AARGB0,f	
	rrf	AARGB1,f	
	rrf	AARGB2,f	
	rrf	AARGB3,f	
	rrf	AARGB4,f	
	rrf	AARGB5,f	
	decfsz	LOOPCOUNT,f	
	goto	CLOOPUM2424	
	return

; ********************************************************************************			
; 32/32 Bit Unsigned Fixed Point Divide 32/32 -> 32.32	
; Use: CALL FXD3232U
;  Input: 32 bit unsigned fixed point dividend in AARGB0, AARGB1,AARGB2,AARGB3	
;       & 32 bit unsigned fixed point divisor in BARGB0, BARGB1, BARGB2, BARGB3	
; 	ms byte xargb0, ls byte xargb30,
; Output: 32 bit unsigned fixed point quotient in AARGB0, AARGB1,AARGB2,AARGB3	
; 32 bit unsigned fixed point remainder in REMB0, REMB1, REMB2, REMB3		
; Result: AARG, REM <-- AARG / BARG		
; Max Timing: 4+1025+2 = 1031 clks		
; Max Timing: 4+981+2 = 987 clks		
; PM: 4+359+1 = 364 DM: 13
; ********************************************************************************			
FXD3232U		
	clrf	REMB0
	clrf	REMB1
	clrf	REMB2
	clrf	REMB3
	call	UDIV3232L
	return
UDIV3232L			
; Max	timing:	24+6*32+31+31+6*32+31+31+6*32+31+31+6*32+31+16=1025clks	
; Min	timing:	24+6*31+30+30+6*31+30+30+6*31+30+30+6*31+30+3=981clks	
; PM:359	dm:13	
	clrf	TEMP
	rlf	AARGB0,w	
	rlf	REMB3,f	
	movf	BARGB3,w	
	subwf	REMB3,f	
	movf	BARGB2,w	
	btfss	STATUS,C	
	incfsz	BARGB2,w	
	subwf	REMB2,f	
	movf	BARGB1,w	
	btfss	STATUS,C	
	incfsz	BARGB1,w	
	subwf	REMB1,f	
	movf	BARGB0,w	
	btfss	STATUS,C	
	incfsz	BARGB0,w	
	subwf	REMB0,f	
	clrw
	btfss	STATUS,C	
	movlw	H'1'	
	subwf	TEMP,f	
	rlf	AARGB0,f	
	movlw	H'7'	
	movwf	LOOPCOUNT	

LOOPU3232A	
	rlf	AARGB0,w	
	rlf	REMB3,f	
	rlf	REMB2,f	
	rlf	REMB1,f	
	rlf	REMB0,f	
	rlf	TEMP,f	
	movf	BARGB3,w	
	btfss	AARGB0,0	
	goto	UADD22LA	
	subwf	REMB3,f	
	movf	BARGB2,w	
	btfss	STATUS,C	
	incfsz	BARGB2,w	
	subwf	REMB2,f	
	movf	BARGB1,w	
	btfss	STATUS,C	
	incfsz	BARGB1,w	
	subwf	REMB1,f	
	movf	BARGB0,w	
	btfss	STATUS,C	
	incfsz	BARGB0,w	
	subwf	REMB0,f	
	clrw
	btfss	STATUS,C	
	movlw	H'1'	
	subwf	TEMP,f	
	goto	UOK22LA	

UADD22LA	
	addwf	REMB3,f	
	movf	BARGB2,w	
	btfsc	STATUS,C	
	incfsz	BARGB2,w	
	addwf	REMB2,f	
	movf	BARGB1,w	
	btfsc	STATUS,C	
	incfsz	BARGB1,w	
	addwf	REMB1,f	
	movf	BARGB0,w	
	btfsc	STATUS,C	
	incfsz	BARGB0,w	
	addwf	REMB0,f	
	clrw
	btfsc	STATUS,C	
	movlw	H'1'	
	addwf	TEMP,f	

UOK22LA	
	rlf	AARGB0,f	
	decfsz	LOOPCOUNT,f	
	goto	LOOPU3232A	
	rlf	AARGB1,w	
	rlf	REMB3,f	
	rlf	REMB2,f	
	rlf	REMB1,f	
	rlf	REMB0,f	
	rlf	TEMP,f	
	movf	BARGB3,w	
	btfss	AARGB0,0	
	goto	UADD22L8	
	subwf	REMB3,f	
	movf	BARGB2,w	
	btfss	STATUS,C	
	incfsz	BARGB2,w	
	subwf	REMB2,f	
	movf	BARGB1,w	
	btfss	STATUS,C	
	incfsz	BARGB1,w	
	subwf	REMB1,f	
	movf	BARGB0,w	
	btfss	STATUS,C	
	incfsz	BARGB0,w	
	subwf	REMB0,f	
	clrw
	btfss	STATUS,C	
	movlw	H'1'	
	subwf	TEMP,f	
	goto	UOK22L8	
			
UADD22L8	
	addwf	REMB3,f	
	movf	BARGB2,w	
	btfsc	STATUS,C	
	incfsz	BARGB2,w	
	addwf	REMB2,f	
	movf	BARGB1,w	
	btfsc	STATUS,C	
	incfsz	BARGB1,w	
	addwf	REMB1,f	
	movf	BARGB0,w	
	btfsc	STATUS,C	
	incfsz	BARGB0,w	
	addwf	REMB0,f	
	clrw
	btfsc	STATUS,C	
	movlw	H'1'	
	addwf	TEMP,f	

UOK22L8	
	rlf	AARGB1,f	
	movlw	H'7'	
	movwf	LOOPCOUNT	

LOOPU3232B	
	rlf	AARGB1,w	
	rlf	REMB3,f	
	rlf	REMB2,f	
	rlf	REMB1,f	
	rlf	REMB0,f	
	rlf	TEMP,f	
	movf	BARGB3,w	
	btfss	AARGB1,0	
	goto	UADD22LB	
	subwf	REMB3,f	
	movf	BARGB2,w	
	btfss	STATUS,C	
	incfsz	BARGB2,w	
	subwf	REMB2,f	
	movf	BARGB1,w	
	btfss	STATUS,C	
	incfsz	BARGB1,w	
	subwf	REMB1,f	
	movf	BARGB0,w	
	btfss	STATUS,C	
	incfsz	BARGB0,w	
	subwf	REMB0,f	
	clrw
	btfss	STATUS,C	
	movlw	H'1'	
	subwf	TEMP,f	
	goto	UOK22LB	

UADD22LB	
	addwf	REMB3,f	
	movf	BARGB2,w	
	btfsc	STATUS,C	
	incfsz	BARGB2,w	
	addwf	REMB2,f	
	movf	BARGB1,w	
	btfsc	STATUS,C	
	incfsz	BARGB1,w	
	addwf	REMB1,f	
	movf	BARGB0,w	
	btfsc	STATUS,C	
	incfsz	BARGB0,w	
	addwf	REMB0,f	
	clrw
	btfsc	STATUS,C	
	movlw	H'1'	
	addwf	TEMP,f	

UOK22LB	
	rlf	AARGB1,f	
	decfsz	LOOPCOUNT,f	
	goto	LOOPU3232B	
	rlf	AARGB2,w	
	rlf	REMB3,f	
	rlf	REMB2,f	
	rlf	REMB1,f	
	rlf	REMB0,f	
	rlf	TEMP,f	
	movf	BARGB3,w	
	btfss	AARGB1,0	
	goto	UADD22L16	
	subwf	REMB3,f	
	movf	BARGB2,w	
	btfss	STATUS,C	
	incfsz	BARGB2,w	
	subwf	REMB2,f	
	movf	BARGB1,w	
	btfss	STATUS,C	
	incfsz	BARGB1,w	
	subwf	REMB1,f	
	movf	BARGB0,w	
	btfss	STATUS,C	
	incfsz	BARGB0,w	
	subwf	REMB0,f	
	clrw	
	btfss	STATUS,C	
	movlw	H'1'	
	subwf	TEMP,f	
	goto	UOK22L16	

UADD22L16	
	addwf	REMB3,f	
	movf	BARGB2,w	
	btfsc	STATUS,C	
	incfsz	BARGB2,w	
	addwf	REMB2,f	
	movf	BARGB1,w	
	btfsc	STATUS,C	
	incfsz	BARGB1,w	
	addwf	REMB1,f	
	movf	BARGB0,w	
	btfsc	STATUS,C	
	incfsz	BARGB0,w	
	addwf	REMB0,f	
	clrw		
	btfsc	STATUS,C	
	movlw	H'1'	
	addwf	TEMP,f	

UOK22L16
	rlf	AARGB2,f	
	movlw	H'7'
	movwf	LOOPCOUNT	

LOOPU3232C
	rlf	AARGB2,w	
	rlf	REMB3,f	
	rlf	REMB2,f	
	rlf	REMB1,f	
	rlf	REMB0,f	
	rlf	TEMP,f	
	movf	BARGB3,w	
	btfss	AARGB2,0	
	goto	UADD22LC	
	subwf	REMB3,f	
	movf	BARGB2,w	
	btfss	STATUS,C	
	incfsz	BARGB2,w	
	subwf	REMB2,f	
	movf	BARGB1,w	
	btfss	STATUS,C	
	incfsz	BARGB1,w	
	subwf	REMB1,f	
	movf	BARGB0,w	
	btfss	STATUS,C	
	incfsz	BARGB0,w	
	subwf	REMB0,f	
	clrw
	btfss	STATUS,C	
	movlw	H'1'	
	subwf	TEMP,f	
	goto	UOK22LC	

UADD22LC	
	addwf	REMB3,f	
	movf	BARGB2,w	
	btfsc	STATUS,C	
	incfsz	BARGB2,w	
	addwf	REMB2,f	
	movf	BARGB1,w	
	btfsc	STATUS,C	
	incfsz	BARGB1,w	
	addwf	REMB1,f	
	movf	BARGB0,w	
	btfsc	STATUS,C	
	incfsz	BARGB0,w	
	addwf	REMB0,f	
	clrw
	btfsc	STATUS,C	
	movlw	H'1'	
	addwf	TEMP,f	

UOK22LC
	rlf	AARGB2,f	
	decfsz	LOOPCOUNT,f	
	goto	LOOPU3232C	
	rlf	AARGB3,w	
	rlf	REMB3,f	
	rlf	REMB2,f	
	rlf	REMB1,f	
	rlf	REMB0,f	
	rlf	TEMP,f	
	movf	BARGB3,w	
	btfss	AARGB2,0	
	goto	UADD22L24	
	subwf	REMB3,f	
	movf	BARGB2,w	
	btfss	STATUS,C	
	incfsz	BARGB2,w	
	subwf	REMB2,f	
	movf	BARGB1,w	
	btfss	STATUS,C	
	incfsz	BARGB1,w	
	subwf	REMB1,f	
	movf	BARGB0,w	
	btfss	STATUS,C	
	incfsz	BARGB0,w	
	subwf	REMB0,f	
	clrw
	btfss	STATUS,C	
	movlw	H'1'	
	subwf	TEMP,f	
	goto	UOK22L24	

UADD22L24	
	addwf	REMB3,f	
	movf	BARGB2,w	
	btfsc	STATUS,C	
	incfsz	BARGB2,w	
	addwf	REMB2,f	
	movf	BARGB1,w	
	btfsc	STATUS,C	
	incfsz	BARGB1,w	
	addwf	REMB1,f	
	movf	BARGB0,w	
	btfsc	STATUS,C	
	incfsz	BARGB0,w	
	addwf	REMB0,f	
	clrw
	btfsc	STATUS,C	
	movlw	H'1'
	addwf	TEMP,f	

UOK22L24
	rlf	AARGB3,f	
	movlw	H'7'	
	movwf	LOOPCOUNT	

LOOPU3232D	
	rlf	AARGB3,w	
	rlf	REMB3,f	
	rlf	REMB2,f	
	rlf	REMB1,f	
	rlf	REMB0,f	
	rlf	TEMP,f	
	movf	BARGB3,w	
	btfss	AARGB3,0	
	goto	UADD22LD	

	subwf	REMB3,f	
	movf	BARGB2,w	
	btfss	STATUS,C	
	incfsz	BARGB2,w	
	subwf	REMB2,f	
	movf	BARGB1,w	
	btfss	STATUS,C	
	incfsz	BARGB1,w	
	subwf	REMB1,f	
	movf	BARGB0,w	
	btfss	STATUS,C	
	incfsz	BARGB0,w	
	subwf	REMB0,f	
	clrw	
	btfss	STATUS,C	
	movlw	H'1'	
	subwf	TEMP,f	
	goto	UOK22LD	

UADD22LD	
	addwf	REMB3,f	
	movf	BARGB2,w	
	btfsc	STATUS,C	
	incfsz	BARGB2,w	
	addwf	REMB2,f	
	movf	BARGB1,w	
	btfsc	STATUS,C	
	incfsz	BARGB1,w	
	addwf	REMB1,f	
	movf	BARGB0,w	
	btfsc	STATUS,C	
	incfsz	BARGB0,w	
	addwf	REMB0,f	
	clrw
	btfsc	STATUS,C	
	movlw	H'1'	
	addwf	TEMP,f	

UOK22LD	
	rlf	AARGB3,f	
	decfsz	LOOPCOUNT,f	
	goto	LOOPU3232D	

	btfsc	AARGB3,0	
	goto	UOK22L	
	movf	BARGB3,w	
	addwf	REMB3,f	
	movf	BARGB2,w	
	btfsc	STATUS,C	
	incfsz	BARGB2,w	
	addwf	REMB2,f	
	movf	BARGB1,w	
	btfsc	STATUS,C	
	incfsz	BARGB1,w	
	addwf	REMB1,f	
	movf	BARGB0,w	
	btfsc	STATUS,C	
	incfsz	BARGB0,w	
	addwf	REMB0,f	
UOK22L
	return

; ********************************************************************************			
; 8 x 8 multiply
; ********************************************************************************			
; input AARGB0 x BARGB0	
; output AARGB0,AARGB1
EIGHTEIGHT:	
	clrf	AARGB1	; clear partial product
UMUL0808L
	movlw	H'08'	
	movwf	LOOPCOUNT	
	movf	AARGB0,w	

LOOPUM0808A
	rrf	BARGB0,f	
	btfsc	STATUS,C	
	goto	LUM0808NAP	
	decfsz	LOOPCOUNT,f	
	goto	LOOPUM0808A	

	clrf	AARGB0	
	retlw	H'00'	
LUM0808NAP
	bcf	STATUS,C	
	goto	LUM0808NA	

LOOPUM0808
	rrf	BARGB0,f	
	btfsc	STATUS,C	
	addwf	AARGB0,f	
LUM0808NA
	rrf	AARGB0,f	
	rrf	AARGB1,f	
	decfsz	LOOPCOUNT,f	
	goto	LOOPUM0808	
	return 		
; ********************************************************************************










; ********************************************************************************
; STRINGS & LOOKUP TABLES
; ********************************************************************************
	org     0x0f00
STR_TABLE:
TABLE_00
	dt	"Fault=          "	; fault-mode  statemachine 0
TABLE_10
	dt	"Burst_Chg       "	; charge mode statemachine 1
TABLE_20
	dt	"Trick_Chg       "	; charge mode statemachine 2
TABLE_30
	dt	"Bulk_Chrg       "	; charge mode statemachine 3
TABLE_40	
	dt	"AbsorbChg       "	; charge mode statemachine 4
TABLE_50
	dt	"Equal_Chg       "	; charge mode statemachine 5
TABLE_60	
	dt	"Float_Chg       "	; charge mode statemachine 6
TABLE_70
	dt	"LowSun!         "	; PauseCharge statemachine 7
TABLE_80
	dt	"DUSK minutes    "	; Dusk-mode   statemachine 8 
TABLE_90
	dt	" (chrg-mode)    "	; menu 1 (now replaced by "charge mode" state)
TABLE_A0
	dt	"SolarCell       "	; menu 2 
TABLE_B0
	dt	"Battery   ", h'00'	; menu 3
	org 0x0fc0			; example of how to terminate a string with $00 + ORG declaration
TABLE_C0
	dt	" (unused)       "	; menu 4 (now replaced by load_state)
TABLE_D0
	dt	"Equalise enabled"	; splash string 0
TABLE_E0
	dt	"SolarCharger EMR"	; splash string 1
TABLE_F0
	dt	"PLN 0v2e 27Mar22"	; splash string 2
STR_TABLE_END:
; ********************************************************************************


; ********************************************************************************			
	end		
; ********************************************************************************





